<?php
// get_subscription.php
// Search subscription_main by mobile_money_number and return policy_number, total_premium, momo_network
// UPDATED: writes audit trail into users_logs
// UPDATED: userindex is ALWAYS set to mobile_money_number (no need to pass userindex)

// =======================
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

// ====== DB CONFIG ======
include("database.php"); // must create $con (mysqli)
if (!isset($con) || !($con instanceof mysqli)) {
  http_response_code(500);
  echo json_encode(["ok" => false, "error" => "DB connection not available. Ensure database.php creates $con (mysqli)."]);
  exit;
}
$con->set_charset("utf8mb4");

/**
 * Insert log into users_logs.
 * Table: users_logs
 *  userindex_log (AUTO_INCREMENT)
 *  userindex
 *  caption
 *  logs_date
 *  path
 *  data_inf
 *
 * Logging should NEVER break the API response.
 */
function write_user_log(mysqli $con, string $userindex, string $caption, string $path, string $data_inf): void
{
  try {
    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";
    $stmt = $con->prepare($sql);
    if (!$stmt) return;

    $stmt->bind_param("ssss", $userindex, $caption, $path, $data_inf);
    $stmt->execute();
    $stmt->close();
  } catch (Throwable $e) {
    // swallow all log errors
  }
}

// ====== Read input: GET or POST ======
$mobile_money_number = '';

if ($_SERVER["REQUEST_METHOD"] === "GET") {
  $mobile_money_number = trim($_GET["mobile_money_number"] ?? "");
} elseif ($_SERVER["REQUEST_METHOD"] === "POST") {
  // supports application/x-www-form-urlencoded or JSON
  $mobile_money_number = trim($_POST["mobile_money_number"] ?? "");

  // JSON fallback
  if ($mobile_money_number === "") {
    $raw = file_get_contents("php://input");
    $json = json_decode($raw, true);
    if (is_array($json)) {
      $mobile_money_number = trim($json["mobile_money_number"] ?? "");
    }
  }
} else {
  http_response_code(405);
  $resp = ["ok" => false, "error" => "Method not allowed. Use GET or POST."];
  echo json_encode($resp);

  // we don't have momo/userindex, use "UNKNOWN"
  write_user_log(
    $con,
    "UNKNOWN",
    "GET_SUBSCRIPTION_METHOD_NOT_ALLOWED",
    ($_SERVER["REQUEST_URI"] ?? "get_subscription.php"),
    json_encode(["request_method" => ($_SERVER["REQUEST_METHOD"] ?? ""), "response" => $resp], JSON_UNESCAPED_UNICODE)
  );

  $con->close();
  exit;
}

if ($mobile_money_number === "") {
  http_response_code(400);
  $resp = ["ok" => false, "error" => "mobile_money_number is required"];
  echo json_encode($resp);

  write_user_log(
    $con,
    "UNKNOWN",
    "GET_SUBSCRIPTION_MISSING_MOMO",
    ($_SERVER["REQUEST_URI"] ?? "get_subscription.php"),
    json_encode(["mobile_money_number" => $mobile_money_number, "response" => $resp], JSON_UNESCAPED_UNICODE)
  );

  $con->close();
  exit;
}

// Optional: normalize (remove spaces, +, etc.)
$mobile_money_number = preg_replace('/\s+/', '', $mobile_money_number);
$mobile_money_number = ltrim($mobile_money_number, '+');

// ✅ YOUR REQUEST:
// userindex is ALWAYS the same as mobile_money_number
$userindex = $mobile_money_number;

// ====== Query ======
$sql = "
  SELECT subscriptions_id,policy_number, total_premium, momo_network
  FROM subscription_main
  WHERE mobile_money_number = ?
  LIMIT 1
";

$stmt = $con->prepare($sql);
if (!$stmt) {
  http_response_code(500);
  $resp = ["ok" => false, "error" => "Prepare failed", "details" => $con->error];
  echo json_encode($resp);

  write_user_log(
    $con,
    $userindex,
    "GET_SUBSCRIPTION_PREPARE_FAILED",
    ($_SERVER["REQUEST_URI"] ?? "get_subscription.php"),
    json_encode(
      ["mobile_money_number" => $mobile_money_number, "userindex_used" => $userindex, "db_error" => $con->error, "response" => $resp],
      JSON_UNESCAPED_UNICODE
    )
  );

  $con->close();
  exit;
}

$stmt->bind_param("s", $mobile_money_number);
if (!$stmt->execute()) {
  http_response_code(500);
  $resp = ["ok" => false, "error" => "Execute failed", "details" => $stmt->error];
  echo json_encode($resp);

  write_user_log(
    $con,
    $userindex,
    "GET_SUBSCRIPTION_EXECUTE_FAILED",
    ($_SERVER["REQUEST_URI"] ?? "get_subscription.php"),
    json_encode(
      ["mobile_money_number" => $mobile_money_number, "userindex_used" => $userindex, "stmt_error" => $stmt->error, "response" => $resp],
      JSON_UNESCAPED_UNICODE
    )
  );

  $stmt->close();
  $con->close();
  exit;
}

$res = $stmt->get_result();
$row = $res ? $res->fetch_assoc() : null;

if (!$row) {
  http_response_code(404);
  $resp = ["ok" => false, "error" => "No subscription found for this mobile_money_number"];
  echo json_encode($resp);

  write_user_log(
    $con,
    $userindex,
    "GET_SUBSCRIPTION_NOT_FOUND",
    ($_SERVER["REQUEST_URI"] ?? "get_subscription.php"),
    json_encode(["mobile_money_number" => $mobile_money_number, "userindex_used" => $userindex, "response" => $resp], JSON_UNESCAPED_UNICODE)
  );

  $stmt->close();
  $con->close();
  exit;
}

// Ensure numeric output for premium
$row["total_premium"] = is_numeric($row["total_premium"]) ? (float)$row["total_premium"] : $row["total_premium"];

$resp = ["ok" => true, "data" => $row];
echo json_encode($resp, JSON_UNESCAPED_UNICODE);

// Log success
write_user_log(
  $con,
  $userindex,
  "GET_SUBSCRIPTION_SUCCESS",
  ($_SERVER["REQUEST_URI"] ?? "get_subscription.php"),
  json_encode(
    [
      "mobile_money_number" => $mobile_money_number,
      "userindex_used" => $userindex,
      "returned" => [     
       "subscriptions_id" => $row["subscriptions_id"] ?? null,
        "policy_number" => $row["policy_number"] ?? null,
        "total_premium" => $row["total_premium"] ?? null,
        "momo_network" => $row["momo_network"] ?? null
      ]
    ],
    JSON_UNESCAPED_UNICODE
  )
);

$stmt->close();
$con->close();