<?php
// policy_search_claims_paid.php
// =====================================================
// Professional Claims Search (Template A) - PHP MySQLi
// 1) Control icon opens Search Result row in popup form (Bootstrap Modal)
// 2) Paging (server-side) + "Paging." aligned right
// 3) Claim Details popup shows Life Name:
//    - If life_type == MAIN -> get main name from policy_details_view by policy_number
//    - If life_type == SECONDARY -> get secondary name from secondary_view by secondary_life_id
// 4) APPROVAL workflow INSIDE Claim Details:
//    - Comment input
//    - Process Approval updates table: claims by claim_index
//      claim_status='APPROVED', approval_comments, approval_datetime, approval_userindex
// 5) PAYMENT workflow INSIDE Claim Details:
//    - Payment comment input
//    - Process Payment updates table: claims by claim_index
//      claim_status='PAID', paid_comments, paid_datetime, paid_userindex
// =====================================================

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include("database.php");
include('log_functions.php');
include("session.php");

date_default_timezone_set("Africa/Accra");

// Helpers
function h($v){ return htmlspecialchars((string)$v, ENT_QUOTES, 'UTF-8'); }
function getv($k, $d=''){ return isset($_REQUEST[$k]) ? trim((string)$_REQUEST[$k]) : $d; }

// Session user
$userindex = isset($_SESSION['userindex']) ? (int)$_SESSION['userindex'] : 0;
$user_sub_priv = htmlspecialchars(filter_var(mysqli_real_escape_string($con, $_SESSION['user_sub_priv'] ?? ''), FILTER_SANITIZE_STRING));

// Messages
$error_message = "";
$success_message = "";

// Action type (search/approve/pay)
$action_type = getv('action_type', '');

// =====================================================
// APPROVAL PROCESSING (POST)
// =====================================================
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action_type === 'approve') {

    if ($userindex <= 0) {
        $error_message = "Session expired. Please login again.";
    } else {

        $claim_index        = (int)getv('claim_index', 0);
        $approval_comments  = trim((string)getv('approval_comments', ''));

        if ($claim_index <= 0) {
            $error_message = "Invalid claim_index.";
        } elseif (strlen($approval_comments) < 3) {
            $error_message = "Approval comment is required (at least 3 characters).";
        } else {

            // Recommended: only approve SUBMITTED claims
            $chk = $con->prepare("SELECT claim_status FROM claims WHERE claim_index = ? LIMIT 1");
            if (!$chk) {
                $error_message = "Prepare failed (check): " . $con->error;
            } else {
                $chk->bind_param("i", $claim_index);

                if (!$chk->execute()) {
                    $error_message = "Execute failed (check): " . $chk->error;
                } else {
                    $res = $chk->get_result()->fetch_assoc();

                    if (!$res) {
                        $error_message = "Claim not found.";
                    } else {

                        $current_status = strtoupper(trim((string)$res['claim_status']));
                        if ($current_status !== 'SUBMITTED') {
                            $error_message = "Cannot approve. Current status is: " . h($current_status);
                        } else {

                            $approval_datetime = date('Y-m-d H:i:s');

                            $sqlU = "UPDATE claims
                                     SET claim_status = 'APPROVED',
                                         approval_comments = ?,
                                         approval_datetime = ?,
                                         approval_userindex = ?
                                     WHERE claim_index = ?";

                            $up = $con->prepare($sqlU);
                            if (!$up) {
                                $error_message = "Prepare failed (update): " . $con->error;
                            } else {

                                $up->bind_param("ssii", $approval_comments, $approval_datetime, $userindex, $claim_index);

                                if (!$up->execute()) {
                                    $error_message = "Update failed: " . $up->error;
                                } else {
                                    $success_message = "Approval saved successfully.";

                                    $data_inf = "Approval Details:\n"
                                              . "Comment: {$approval_comments}\n"
                                              . "Approval DateTime: {$approval_datetime}\n"
                                              . "Approved By (userindex): {$userindex}\n"
                                              . "Claim Index: {$claim_index}";

                                    // Log successful operation
                                    $page_path = 'policy_search_claims_paid.php';
                                    $reasons = 'Claim successfully Approved.';
                                    log_failed_login($con, $userindex, $page_path, $reasons, $data_inf);
                                }

                                $up->close();
                            }
                        }
                    }
                }

                $chk->close();
            }
        }
    }
}

// =====================================================
// PAYMENT PROCESSING (POST)
// =====================================================
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $action_type === 'pay') {

    if ($userindex <= 0) {
        $error_message = "Session expired. Please login again.";
    } else {

        $claim_index    = (int)getv('claim_index', 0);
        $paid_comments  = trim((string)getv('paid_comments', ''));

        if ($claim_index <= 0) {
            $error_message = "Invalid claim_index.";
        } elseif (strlen($paid_comments) < 3) {
            $error_message = "Payment comment is required (at least 3 characters).";
        } else {

            // Recommended: only pay APPROVED claims
            $chk = $con->prepare("SELECT claim_status FROM claims WHERE claim_index = ? LIMIT 1");
            if (!$chk) {
                $error_message = "Prepare failed (check): " . $con->error;
            } else {
                $chk->bind_param("i", $claim_index);

                if (!$chk->execute()) {
                    $error_message = "Execute failed (check): " . $chk->error;
                } else {
                    $res = $chk->get_result()->fetch_assoc();

                    if (!$res) {
                        $error_message = "Claim not found.";
                    } else {

                        $current_status = strtoupper(trim((string)$res['claim_status']));
                        if ($current_status !== 'APPROVED') {
                            $error_message = "Cannot process payment. Current status is: " . h($current_status);
                        } else {

                            $paid_datetime = date('Y-m-d H:i:s');

                            $sqlU = "UPDATE claims
                                     SET claim_status   = 'PAID',
                                         paid_comments  = ?,
                                         paid_datetime  = ?,
                                         paid_userindex = ?
                                     WHERE claim_index = ?";

                            $up = $con->prepare($sqlU);
                            if (!$up) {
                                $error_message = "Prepare failed (update): " . $con->error;
                            } else {

                                $up->bind_param("ssii", $paid_comments, $paid_datetime, $userindex, $claim_index);

                                if (!$up->execute()) {
                                    $error_message = "Update failed: " . $up->error;
                                } else {

                                    $success_message = "Payment processed successfully. Claim marked as PAID.";

                                    $data_inf = "Payment Details:\n"
                                              . "Comment: {$paid_comments}\n"
                                              . "Paid DateTime: {$paid_datetime}\n"
                                              . "Paid By (userindex): {$userindex}\n"
                                              . "Claim Index: {$claim_index}";

                                    $page_path = 'policy_search_claims_paid.php';
                                    $reasons   = 'Claim successfully PAID.';
                                    log_failed_login($con, $userindex, $page_path, $reasons, $data_inf);
                                }

                                $up->close();
                            }
                        }
                    }
                }

                $chk->close();
            }
        }
    }
}

// =====================================================
// SEARCH FILTERS / INPUTS
// =====================================================
$search_claim_id      = getv('claim_id', '');
$search_policy_number = getv('policy_number', '');
$start_date           = getv('start_date', date('Y-m-01'));
$end_date             = getv('end_date', date('Y-m-d'));
$status_filter        = strtoupper(getv('claim_status', 'ALL')); // ALL/SUBMITTED/APPROVED/PAID

// Search performed rules:
// - when user hits Search (POST action_type=search)
// - OR when user approves (POST action_type=approve) and we want to reload results
// - OR when user pays (POST action_type=pay) and we want to reload results
$search_performed = false;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($action_type === 'search' || $action_type === 'approve' || $action_type === 'pay')) {
    $search_performed = true;
}
if ($_SERVER['REQUEST_METHOD'] === 'GET' && (isset($_GET['page']) || isset($_GET['page_size']))) {
    $search_performed = true;
}

// Paging (server side)
$page     = (int)getv('page', 1);
$pageSize = (int)getv('page_size', 10);
if ($page < 1) $page = 1;
if ($pageSize < 5) $pageSize = 5;
if ($pageSize > 100) $pageSize = 100;

$offset = ($page - 1) * $pageSize;

$allowed_status = ['ALL','SUBMITTED','APPROVED','PAID'];
if (!in_array($status_filter, $allowed_status, true)) $status_filter = 'ALL';

// Build WHERE (prepared)
$where = "WHERE 1=1";
$types = "";
$params = [];

if ($search_claim_id !== "") {
    $where .= " AND c.claim_id LIKE ?";
    $types .= "s";
    $params[] = "%{$search_claim_id}%";
}
if ($search_policy_number !== "") {
    $where .= " AND c.policy_number LIKE ?";
    $types .= "s";
    $params[] = "%{$search_policy_number}%";
}
if ($start_date !== "" && $end_date !== "") {
    $where .= " AND DATE(c.claims_date) BETWEEN ? AND ?";
    $types .= "ss";
    $params[] = $start_date;
    $params[] = $end_date;
} elseif ($start_date !== "") {
    $where .= " AND DATE(c.claims_date) >= ?";
    $types .= "s";
    $params[] = $start_date;
} elseif ($end_date !== "") {
    $where .= " AND DATE(c.claims_date) <= ?";
    $types .= "s";
    $params[] = $end_date;
}

if ($status_filter !== "ALL") {
    $where .= " AND c.claim_status = ?";
    $types .= "s";
    $params[] = $status_filter;
}

// Custom status sorting: SUBMITTED -> APPROVED -> PAID
$orderBy = "
ORDER BY
  CASE c.claim_status
    WHEN 'SUBMITTED' THEN 1
    WHEN 'APPROVED'  THEN 2
    WHEN 'PAID'      THEN 3
    ELSE 99
  END ASC,
  c.claims_date DESC,
  c.claim_index DESC
";

$search_results = [];
$total_rows = 0;
$total_pages = 0;

// COUNT for paging
if ($search_performed && $error_message === "") {
    $countSql = "SELECT COUNT(*) AS cnt FROM claims c {$where}";
    $stmt = $con->prepare($countSql);
    if (!$stmt) {
        $error_message = "Prepare failed (count): " . $con->error;
    } else {
        if ($types !== "") $stmt->bind_param($types, ...$params);

        if (!$stmt->execute()) {
            $error_message = "Execute failed (count): " . $stmt->error;
        } else {
            $row = $stmt->get_result()->fetch_assoc();
            $total_rows = (int)($row['cnt'] ?? 0);
            $total_pages = (int)ceil($total_rows / $pageSize);
            if ($total_pages < 1) $total_pages = 1;
            if ($page > $total_pages) $page = $total_pages;
            $offset = ($page - 1) * $pageSize;
        }
        $stmt->close();
    }
}

// Query (results) + join to get life_name
$sql = "
SELECT
    c.claim_index,
    c.claim_id,
    c.policy_number,
    c.policyholder_id,
    c.life_type,
    c.secondary_life_id,
    c.claim_type,
    c.critical_illness_type,
    c.event_date,
    c.reported_date,
    c.sum_assured,
    c.approved_amount,
    c.claim_status,
    c.claims_date,

    CASE
      WHEN UPPER(TRIM(c.life_type)) = 'MAIN' THEN pdv.full_name
      WHEN UPPER(TRIM(c.life_type)) = 'SECONDARY' THEN sv.full_name
      ELSE NULL
    END AS life_name
FROM claims c
LEFT JOIN policy_details_view pdv
  ON pdv.policy_number = c.policy_number
LEFT JOIN secondary_view sv
  ON sv.secondary_life_id = c.secondary_life_id
{$where}
{$orderBy}
LIMIT ? OFFSET ?
";

// Run results
if ($search_performed && $error_message === "") {
    $stmt = $con->prepare($sql);
    if (!$stmt) {
        $error_message = "Prepare failed: " . $con->error;
    } else {
        $types2 = $types . "ii";
        $params2 = $params;
        $params2[] = $pageSize;
        $params2[] = $offset;

        $stmt->bind_param($types2, ...$params2);

        if (!$stmt->execute()) {
            $error_message = "Execute failed: " . $stmt->error;
        } else {
            $res = $stmt->get_result();
            while ($row = $res->fetch_assoc()) $search_results[] = $row;
        }
        $stmt->close();
    }
}

// Stats (for current filters)
$date_stats = [
    'earliest_date' => null,
    'latest_date' => null,
    'total_claims' => 0,
    'avg_approval' => 0,
    'total_approved' => 0
];

if ($search_performed && $error_message === "") {
    $statsSql = "
      SELECT
        MIN(c.claims_date) AS earliest_date,
        MAX(c.claims_date) AS latest_date,
        COUNT(*) AS total_claims,
        AVG(c.approved_amount) AS avg_approval,
        SUM(c.approved_amount) AS total_approved
      FROM claims c
      {$where}
    ";
    $st = $con->prepare($statsSql);
    if ($st) {
        if ($types !== "") $st->bind_param($types, ...$params);

        if ($st->execute()) {
            $r = $st->get_result()->fetch_assoc();
            if ($r) $date_stats = $r;
        }
        $st->close();
    }
}

// Build querystring for paging links (keep filters)
function buildPagingUrl(array $overrides = []) {
    $q = $_REQUEST;
    foreach ($overrides as $k => $v) $q[$k] = $v;
    return "policy_search_claims_paid.php?" . http_build_query($q);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Claims Search System</title>

  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">

  <style>
    body{ background:#f5f5f5; font-family:'Segoe UI',Tahoma,Geneva,Verdana,sans-serif; }
    .card{ border-radius:14px; box-shadow:0 8px 18px rgba(0,0,0,.10); border:none; overflow:hidden; }
    .card-header{ background:#6ec1e4; color:#fff; text-align:center; padding:18px; }
    .btn-search{ background:#1976d2; border:none; color:#fff; padding:10px 28px; font-weight:900; border-radius:10px; }
    .btn-search:hover{ background:#125aa0; }
    .btn-reset{ background:#6c757d; border:none; color:#fff; padding:10px 18px; font-weight:900; border-radius:10px; text-decoration:none; display:inline-block; }
    .table-container{ max-height:520px; overflow:auto; border:1px solid #dee2e6; border-radius:10px; }
    .table thead th{ background:#6ec1e4; color:#fff; position:sticky; top:0; z-index:10; }
    .amount{ font-weight:900; color:#2c3e50; }
    .pill{ display:inline-flex; gap:8px; align-items:center; padding:8px 10px; border-radius:999px; background:#f1f7ff; border:1px solid #d6e8ff; font-weight:800; font-size:12px; }
    .st-submitted{ background:#e9f2ff; color:#0b4fd8; font-weight:900; padding:4px 10px; border-radius:999px; border:1px solid #cfe3ff; }
    .st-approved{ background:#e9fbf1; color:#137a3a; font-weight:900; padding:4px 10px; border-radius:999px; border:1px solid #c8f0d7; }
    .st-paid{ background:#fff7e6; color:#8a5a00; font-weight:900; padding:4px 10px; border-radius:999px; border:1px solid #ffe1a8; }
    .st-other{ background:#f2f2f2; color:#444; font-weight:900; padding:4px 10px; border-radius:999px; border:1px solid #ddd; }

    /* Paging (align right + label "Paging.") */
    .paging-wrap{ display:flex; justify-content:flex-end; align-items:center; gap:10px; margin-top:12px; flex-wrap:wrap; }
    .paging-label{ font-weight:900; color:#0b4fd8; }
    .paging a, .paging span{
      display:inline-block; padding:6px 10px; border-radius:10px; border:1px solid #d6e8ff;
      background:#f1f7ff; font-weight:900; text-decoration:none; color:#0b4fd8;
    }
    .paging .active{ background:#1976d2; color:#fff; border-color:#1976d2; }
    .paging .disabled{ opacity:.45; pointer-events:none; }
  </style>
</head>

<?php include 'main_menu_app.php'; ?>

<style>
  body {
    background-image: url('images/ayo_banner4.png');
    background-size: cover;
    background-position: center;
    background-repeat: no-repeat;
    background-attachment: fixed;
    opacity: 0.9;
  }
  body::before {
    content:"";
    position:fixed; top:0; left:0; width:100%; height:100%;
    background-image:url('images/ayo_banner4.png');
    background-size:cover;
    background-position:center;
    background-repeat:no-repeat;
    background-attachment:fixed;
    opacity:0.5;
    z-index:-1;
  }
</style>

<body>
<div class="content">
  <div class="container-fluid">
    <div class="row mt-4">
      <div class="col-md-12">

        <div class="card">
          <div class="card-header">
            <h4 class="card-title" style="margin:0; font-weight:900;">
              <i class="fas fa-search"></i> Policy Search To Process Claims
            </h4>
            <p class="card-category" style="margin:6px 0 0; opacity:.95;">
              Search by Claim ID, Policy Number • Filter by Status & Date Range
            </p>
          </div>

          <div class="card-body">

            <?php if ($error_message !== ""): ?>
              <div class="alert alert-danger" style="border-radius:12px;">
                <b>Error:</b> <?php echo h($error_message); ?>
              </div>
            <?php endif; ?>

            <?php if ($success_message !== ""): ?>
              <div class="alert alert-success" style="border-radius:12px;">
                <b>Success:</b> <?php echo h($success_message); ?>
              </div>
            <?php endif; ?>

            <!-- SEARCH FORM -->
            <form id="searchForm" method="POST" action="">
              <input type="hidden" name="action_type" value="search">
              <input type="hidden" name="page" value="1">
              <div class="row mb-3">
                <div class="col-md-3">
                  <label class="form-label fw-bold"><i class="fas fa-file-alt"></i> Claim ID</label>
                  <input type="text" class="form-control" name="claim_id"
                         value="<?php echo h($search_claim_id); ?>"
                         placeholder="e.g. CL260100003">
                </div>

                <div class="col-md-3">
                  <label class="form-label fw-bold"><i class="fas fa-file-contract"></i> Policy Number</label>
                  <input type="text" class="form-control" name="policy_number"
                         value="<?php echo h($search_policy_number); ?>"
                         placeholder="e.g. ABR01260000011">
                </div>

                <div class="col-md-2">
                  <label class="form-label fw-bold"><i class="fas fa-filter"></i> Claim Status</label>
                  <select class="form-control" name="claim_status">
                    <?php foreach ($allowed_status as $st): ?>
                      <option value="<?php echo h($st); ?>" <?php echo ($status_filter===$st?'selected':''); ?>>
                        <?php echo h($st); ?>
                      </option>
                    <?php endforeach; ?>
                  </select>
                  <small class="text-muted">Sorted: SUBMITTED → APPROVED → PAID</small>
                </div>

                <div class="col-md-2">
                  <label class="form-label fw-bold"><i class="far fa-calendar"></i> From</label>
                  <input type="date" class="form-control" name="start_date" value="<?php echo h($start_date); ?>">
                </div>

                <div class="col-md-2">
                  <label class="form-label fw-bold"><i class="far fa-calendar"></i> To</label>
                  <input type="date" class="form-control" name="end_date" value="<?php echo h($end_date); ?>">
                </div>
              </div>

              <div class="row">
                <div class="col-md-12 d-flex justify-content-end gap-2">
                  <button type="submit" class="btn-search">
                    <i class="fas fa-search"></i> Search
                  </button>
                  <a href="policy_search_claims_paid.php" class="btn-reset">
                    <i class="fas fa-redo"></i> Reset
                  </a>
                </div>
              </div>
            </form>

            <?php if ($search_performed): ?>
              <div class="row mt-3">
                <div class="col-md-12 d-flex gap-2 flex-wrap">
                  <span class="pill"><i class="fas fa-list"></i> Results: <?php echo (int)($date_stats['total_claims'] ?? 0); ?></span>
                  <span class="pill"><i class="fas fa-calendar"></i>
                    <?php
                      if (!empty($date_stats['earliest_date']) && !empty($date_stats['latest_date'])) {
                        echo h(date('M d, Y', strtotime($date_stats['earliest_date']))) . " to " . h(date('M d, Y', strtotime($date_stats['latest_date'])));
                      } else {
                        echo "All Dates";
                      }
                    ?>
                  </span>
                  <span class="pill"><i class="fas fa-coins"></i> Total Approved: <?php echo number_format((float)($date_stats['total_approved'] ?? 0), 2); ?></span>
                  <span class="pill"><i class="fas fa-chart-line"></i> Avg Approval: <?php echo number_format((float)($date_stats['avg_approval'] ?? 0), 2); ?></span>
                </div>
              </div>

              <div class="card mt-3">
                <div class="card-body">
                  <div class="d-flex justify-content-between align-items-center mb-2">
                    <h6 class="mb-0 fw-bold"><i class="fas fa-list-alt"></i> Search Results</h6>
                    <span class="badge bg-primary"><?php echo (int)$total_rows; ?> total</span>
                  </div>

                  <?php if (empty($search_results)): ?>
                    <div class="alert alert-warning text-center" style="border-radius:12px;">
                      <i class="fas fa-exclamation-triangle"></i> No claims found matching your filters.
                    </div>
                  <?php else: ?>
                    <div class="table-responsive table-container">
                      <table class="table table-hover" id="claimsTable">
                        <thead>
                          <tr>
                            <th>#</th>
                            <th>Claim ID</th>
                            <th>Policy No.</th>
                            <th>Life Name</th>
                            <th>Type</th>
                            <th>Event Date</th>
                            <th>Claims Date</th>
                            <th>S.A</th>
                            <th>Approved Amt</th>
                            <th>Status</th>
                            <th>Life</th>
                            <th>Control</th>
                          </tr>
                        </thead>
                        <tbody>
                          <?php foreach ($search_results as $i => $claim): ?>
                            <?php
                              $st = strtoupper(trim((string)$claim['claim_status']));
                              $cls = "st-other";
                              if ($st === "SUBMITTED") $cls = "st-submitted";
                              if ($st === "APPROVED")  $cls = "st-approved";
                              if ($st === "PAID")      $cls = "st-paid";

                              $modalData = json_encode([
                                "claim_index" => $claim["claim_index"] ?? "",
                                "claim_id" => $claim["claim_id"] ?? "",
                                "policy_number" => $claim["policy_number"] ?? "",
                                "claim_status" => $claim["claim_status"] ?? "",
                                "life_type" => $claim["life_type"] ?? "",
                                "secondary_life_id" => $claim["secondary_life_id"] ?? "",
                                "life_name" => $claim["life_name"] ?? "",
                                "claim_type" => $claim["claim_type"] ?? "",
                                "critical_illness_type" => $claim["critical_illness_type"] ?? "",
                                "event_date" => $claim["event_date"] ?? "",
                                "reported_date" => $claim["reported_date"] ?? "",
                                "sum_assured" => $claim["sum_assured"] ?? "",
                                "approved_amount" => $claim["approved_amount"] ?? "",
                                "claims_date" => $claim["claims_date"] ?? "",
                              ], JSON_HEX_TAG|JSON_HEX_APOS|JSON_HEX_AMP|JSON_HEX_QUOT);

                              $rowNo = $offset + $i + 1;
                            ?>
                            <tr>
                              <td><?php echo (int)$rowNo; ?></td>
                              <td><b class="text-primary"><?php echo h($claim['claim_id']); ?></b></td>
                              <td><span class="text-info"><?php echo h($claim['policy_number']); ?></span></td>
                              <td><?php echo h($claim['life_name'] ?: '-'); ?></td>
                              <td>
                                <?php echo h($claim['claim_type']); ?>
                                <?php if (!empty($claim['critical_illness_type'])): ?>
                                  <br><small class="text-muted"><?php echo h($claim['critical_illness_type']); ?></small>
                                <?php endif; ?>
                              </td>
                              <td><?php echo h($claim['event_date']); ?></td>
                              <td><?php echo h($claim['claims_date']); ?></td>
                              <td class="amount"><?php echo number_format((float)$claim['sum_assured'], 2); ?></td>
                              <td class="amount text-success"><?php echo number_format((float)$claim['approved_amount'], 2); ?></td>
                              <td><span class="<?php echo h($cls); ?>"><?php echo h($st ?: 'N/A'); ?></span></td>
                              <td>
                                <?php echo h($claim['life_type']); ?>
                                <?php if (!empty($claim['secondary_life_id']) && (int)$claim['secondary_life_id'] > 0): ?>
                                  <br><small class="text-muted">Secondary ID: <?php echo (int)$claim['secondary_life_id']; ?></small>
                                <?php endif; ?>
                              </td>

                              <!-- CONTROL: opens popup form -->
                              <td style="white-space:nowrap;">
                                <button
                                  type="button"
                                  class="btn btn-success btn-sm"
                                  title="View / Process"
                                  style="border-radius:10px; font-weight:900;"
                                  onclick='openClaimModal(<?php echo $modalData; ?>)'>
                                  <i class="fas fa-check-circle"></i>
                                </button>
                              </td>
                            </tr>
                          <?php endforeach; ?>
                        </tbody>
                      </table>
                    </div>

                    <!-- Paging -->
                    <div class="paging-wrap">
                      <span class="paging-label">Paging.</span>

                      <div class="paging d-flex gap-2 flex-wrap">
                        <?php
                          $prev = $page - 1;
                          $next = $page + 1;

                          $prevClass = ($page <= 1) ? "disabled" : "";
                          $nextClass = ($page >= $total_pages) ? "disabled" : "";

                          echo '<a class="'.$prevClass.'" href="'.h(buildPagingUrl(['page'=>$prev])).'">← Prev</a>';

                          $startWin = max(1, $page - 3);
                          $endWin   = min($total_pages, $page + 3);

                          if ($startWin > 1) {
                              echo '<a href="'.h(buildPagingUrl(['page'=>1])).'">1</a>';
                              if ($startWin > 2) echo '<span class="disabled">...</span>';
                          }

                          for ($p=$startWin; $p<=$endWin; $p++) {
                              if ($p == $page) {
                                  echo '<span class="active">'.$p.'</span>';
                              } else {
                                  echo '<a href="'.h(buildPagingUrl(['page'=>$p])).'">'.$p.'</a>';
                              }
                          }

                          if ($endWin < $total_pages) {
                              if ($endWin < $total_pages - 1) echo '<span class="disabled">...</span>';
                              echo '<a href="'.h(buildPagingUrl(['page'=>$total_pages])).'">'.$total_pages.'</a>';
                          }

                          echo '<a class="'.$nextClass.'" href="'.h(buildPagingUrl(['page'=>$next])).'">Next →</a>';
                        ?>
                      </div>

                      <!-- Page size -->
                      <form method="GET" action="policy_search_claims_paid.php" class="d-flex gap-2 align-items-center">
                        <?php
                          foreach ($_REQUEST as $k=>$v) {
                              if ($k === 'page_size') continue;
                              echo '<input type="hidden" name="'.h($k).'" value="'.h($v).'">';
                          }
                        ?>
                        <label class="fw-bold mb-0">Per page</label>
                        <select name="page_size" class="form-control" style="width:110px;" onchange="this.form.submit()">
                          <?php foreach ([10,20,30,50,100] as $ps): ?>
                            <option value="<?php echo $ps; ?>" <?php echo ($pageSize===$ps?'selected':''); ?>><?php echo $ps; ?></option>
                          <?php endforeach; ?>
                        </select>
                      </form>
                    </div>
                  <?php endif; ?>

                </div>
              </div>
            <?php endif; ?>

          </div>
        </div>

      </div>
    </div>
  </div>
</div>

<!-- Modal: Claim Details -->
<div class="modal fade" id="claimModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content" style="border-radius:14px;">
      <div class="modal-header" style="background:#6ec1e4; color:#fff;">
        <h5 class="modal-title">
          <i class="fas fa-check-circle"></i> Claim Details
        </h5>
        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>

      <div class="modal-body">
        <div class="row g-3">

          <!-- Hidden claim_index -->
          <input type="hidden" id="m_claim_index">

          <div class="col-md-4">
            <label class="fw-bold">Claim ID</label>
            <input class="form-control" id="m_claim_id" readonly>
          </div>
          <div class="col-md-4">
            <label class="fw-bold">Policy Number</label>
            <input class="form-control" id="m_policy_number" readonly>
          </div>
          <div class="col-md-4">
            <label class="fw-bold">Status</label>
            <input class="form-control" id="m_claim_status" readonly>
          </div>

          <div class="col-md-6">
            <label class="fw-bold">Life Type</label>
            <input class="form-control" id="m_life_type" readonly>
          </div>
          <div class="col-md-6">
            <label class="fw-bold">Life Name</label>
            <input class="form-control" id="m_life_name" readonly>
            <small class="text-muted">MAIN: from policy_details • SECONDARY: from secondary_details</small>
          </div>

          <div class="col-md-4">
            <label class="fw-bold">Secondary Life ID</label>
            <input class="form-control" id="m_secondary_life_id" readonly>
          </div>
          <div class="col-md-4">
            <label class="fw-bold">Claim Type</label>
            <input class="form-control" id="m_claim_type" readonly>
          </div>
          <div class="col-md-4">
            <label class="fw-bold">Critical Illness Type</label>
            <input class="form-control" id="m_ci_type" readonly>
          </div>

          <div class="col-md-4">
            <label class="fw-bold">Event Date</label>
            <input class="form-control" id="m_event_date" readonly>
          </div>
          <div class="col-md-4">
            <label class="fw-bold">Reported Date</label>
            <input class="form-control" id="m_reported_date" readonly>
          </div>
          <div class="col-md-4">
            <label class="fw-bold">Claims Date</label>
            <input class="form-control" id="m_claims_date" readonly>
          </div>

          <div class="col-md-6">
            <label class="fw-bold">Sum Assured</label>
            <input class="form-control" id="m_sum_assured" readonly>
          </div>
          <div class="col-md-6">
            <label class="fw-bold">Approved Amount</label>
            <input class="form-control" id="m_approved_amount" readonly>
          </div>

          <!-- Payment comment -->
          <div class="col-md-12 mt-2">
            <label class="fw-bold">Payment Comment</label>
            <textarea class="form-control" id="m_paid_comment" rows="3"
              placeholder="Add payment comment..."></textarea>
            <small class="text-muted">Required to process payment.</small>
          </div>

        </div>

        <div class="alert alert-info mt-3 mb-0" style="border-radius:12px;">
          Click <b>Process Payment</b> for this claim (Only APPROVED claims can be paid).
        </div>
      </div>

      <div class="modal-footer">
        <button type="button" id="m_process_btn" class="btn btn-success" style="border-radius:10px; font-weight:900;">
          <i class="fas fa-cogs"></i> Process Payment
        </button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" style="border-radius:10px; font-weight:900;">
          Close
        </button>
      </div>
    </div>
  </div>
</div>

<!-- Bootstrap JS -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>

<script>
let _currentClaimStatus = '';

function openClaimModal(data){
  document.getElementById('m_claim_index').value       = data.claim_index || '';
  document.getElementById('m_claim_id').value          = data.claim_id || '';
  document.getElementById('m_policy_number').value     = data.policy_number || '';
  document.getElementById('m_claim_status').value      = data.claim_status || '';
  document.getElementById('m_life_type').value         = data.life_type || '';
  document.getElementById('m_secondary_life_id').value = data.secondary_life_id || '';
  document.getElementById('m_life_name').value         = data.life_name || '';
  document.getElementById('m_claim_type').value        = data.claim_type || '';
  document.getElementById('m_ci_type').value           = data.critical_illness_type || '';
  document.getElementById('m_event_date').value        = data.event_date || '';
  document.getElementById('m_reported_date').value     = data.reported_date || '';
  document.getElementById('m_claims_date').value       = data.claims_date || '';
  document.getElementById('m_sum_assured').value       = data.sum_assured || '';
  document.getElementById('m_approved_amount').value   = data.approved_amount || '';

  // reset payment comment
  document.getElementById('m_paid_comment').value  = '';

  _currentClaimStatus = (data.claim_status || '').toString().toUpperCase().trim();

  // Only allow payment for APPROVED
  document.getElementById('m_process_btn').disabled = (_currentClaimStatus !== 'APPROVED');

  const modal = new bootstrap.Modal(document.getElementById('claimModal'));
  modal.show();
}

document.getElementById('m_process_btn').addEventListener('click', function () {
  const claimIndex = document.getElementById('m_claim_index').value;
  const comment = document.getElementById('m_paid_comment').value.trim();

  if (!claimIndex) { alert("Missing claim_index."); return; }
  if (comment.length < 3) { alert("Please enter a payment comment (at least 3 characters)."); return; }
  if (_currentClaimStatus !== 'APPROVED') { alert("Only APPROVED claims can be paid."); return; }

  // Keep current search filters (so results reload after payment)
  const searchForm = document.getElementById('searchForm');

  const f = document.createElement('form');
  f.method = 'POST';
  f.action = 'policy_search_claims_paid.php';

  const addHidden = (name, value) => {
    const i = document.createElement('input');
    i.type = 'hidden';
    i.name = name;
    i.value = value ?? '';
    f.appendChild(i);
  };

  addHidden('action_type', 'pay');
  addHidden('claim_index', claimIndex);
  addHidden('paid_comments', comment);

  // Copy search filters
  const fields = ['claim_id','policy_number','claim_status','start_date','end_date','page','page_size'];
  fields.forEach(fn => {
    const el = searchForm.querySelector(`[name="${fn}"]`);
    if (el) addHidden(fn, el.value);
  });

  document.body.appendChild(f);
  f.submit();
});
</script>

</body>
</html>
