<?php
/**
 * policy_payment_upload.php
 * -------------------------
 * Adds a CLEAR/RESET button to clear messages + summary and allow new upload.
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include("database.php");
include("session.php");

date_default_timezone_set("Africa/Accra");

$userindex  = isset($_SESSION['userindex']) ? (int)$_SESSION['userindex'] : 0;
$created_by = $userindex;

$success_message = "";
$error_message   = "";
$upload_summary  = null;

// ------------------------------
// ✅ CLEAR BUTTON (NEW)
// ------------------------------
if (isset($_GET['clear']) && $_GET['clear'] === '1') {
    // clear all UI state
    $success_message = "";
    $error_message   = "";
    $upload_summary  = null;

    // redirect to remove ?clear=1 from URL (clean reload)
    header("Location: " . strtok($_SERVER["REQUEST_URI"], '?'));
    exit;
}

// ------------------------------
// Helpers
// ------------------------------
function normalizePolicyNumber(string $v): string {
    $v = trim($v);
    $v = preg_replace('/\s+/', '', $v);
    return $v;
}

function normalizeAmount($v): ?float {
    if ($v === null) return null;
    $v = trim((string)$v);
    if ($v === '') return null;
    $v = preg_replace('/[^\d\.\-]/', '', $v);
    if ($v === '' || !is_numeric($v)) return null;
    return (float)$v;
}

function normalizeDate(string $v): ?string {
    $v = trim($v);
    if ($v === '') return null;

    if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $v)) return $v;

    $v2 = str_replace(['.', '\\'], ['/', '/'], $v);
    $formats = ['d/m/Y', 'm/d/Y', 'd-m-Y', 'm-d-Y', 'd/m/y', 'm/d/y', 'd-m-y', 'm-d-y', 'Y/m/d', 'Y-m-d'];

    foreach ($formats as $f) {
        $dt = DateTime::createFromFormat($f, $v2);
        if ($dt && $dt->format($f) === $v2) return $dt->format('Y-m-d');
    }

    $ts = strtotime($v);
    if ($ts !== false) return date('Y-m-d', $ts);

    return null;
}

function readCsvRows(string $filepath): array {
    $rows = [];
    if (($handle = fopen($filepath, "r")) === false) return $rows;

    $header = null;
    while (($data = fgetcsv($handle)) !== false) {
        if ($header === null) {
            $header = array_map(function($h){
                $h = strtolower(trim((string)$h));
                $h = str_replace([' ', '-'], '_', $h);
                return $h;
            }, $data);
            continue;
        }

        $allEmpty = true;
        foreach ($data as $c) {
            if (trim((string)$c) !== '') { $allEmpty = false; break; }
        }
        if ($allEmpty) continue;

        $row = [];
        foreach ($header as $i => $key) $row[$key] = $data[$i] ?? '';
        $rows[] = $row;
    }

    fclose($handle);
    return $rows;
}

function readExcelRows(string $filepath): array {
    $autoload = __DIR__ . "/vendor/autoload.php";
    if (!file_exists($autoload)) {
        throw new Exception("Excel upload requires PhpSpreadsheet. Run: composer require phpoffice/phpspreadsheet");
    }
    require_once $autoload;

    $rows = [];
    $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($filepath);
    $reader->setReadDataOnly(true);

    $spreadsheet = $reader->load($filepath);
    $sheet = $spreadsheet->getActiveSheet();
    $data = $sheet->toArray(null, true, true, true);

    if (!$data || count($data) < 2) return $rows;

    $firstRow = array_shift($data);
    $header = [];
    foreach ($firstRow as $col => $h) {
        $key = strtolower(trim((string)$h));
        $key = str_replace([' ', '-'], '_', $key);
        $header[$col] = $key;
    }

    foreach ($data as $r) {
        $row = [];
        $allEmpty = true;
        foreach ($header as $col => $key) {
            $val = isset($r[$col]) ? (string)$r[$col] : '';
            if (trim($val) !== '') $allEmpty = false;
            $row[$key] = $val;
        }
        if ($allEmpty) continue;
        $rows[] = $row;
    }

    return $rows;
}

function fetchPolicySet(mysqli $con): array {
    $set = [];
    $sql = "SELECT policy_number FROM policy_holders";
    $res = $con->query($sql);
    if ($res) {
        while ($r = $res->fetch_assoc()) {
            $pn = normalizePolicyNumber($r['policy_number'] ?? '');
            if ($pn !== '') $set[$pn] = true;
        }
        $res->free();
    }
    return $set;
}

function insertUploadSummary(
    mysqli $con,
    string $file_name,
    int $inserted,
    int $skipped,
    string $batch_id,
    string $reference,
    int $userindex
): bool {
    $sql = "
        INSERT INTO premium_upload_summary
        (file_name, inserted_count, skipped_count, payment_batch_id, payment_reference, userindex, created_at)
        VALUES (?, ?, ?, ?, ?, ?, NOW())
    ";
    $stmt = $con->prepare($sql);
    if (!$stmt) return false;

    $stmt->bind_param("siissi", $file_name, $inserted, $skipped, $batch_id, $reference, $userindex);
    $ok = $stmt->execute();
    $stmt->close();
    return $ok;
}

// ------------------------------
// Handle Upload
// ------------------------------
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['upload_premiums'])) {
    try {
        if ($userindex <= 0) throw new Exception("Session user not found. Please log in again.");

        if (!isset($_FILES['premium_file']) || $_FILES['premium_file']['error'] !== UPLOAD_ERR_OK) {
            throw new Exception("Please select a valid CSV/Excel file to upload.");
        }

        $payment_reference = trim($_POST['payment_reference'] ?? '');
        $payment_batch_id  = trim($_POST['payment_batch_id'] ?? '');

        if ($payment_batch_id === '') {
            $payment_batch_id = "BATCH-" . date('Ymd-His') . "-" . $userindex;
        }

        $tmp  = $_FILES['premium_file']['tmp_name'];
        $name = $_FILES['premium_file']['name'] ?? 'upload';
        $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));

        if (!in_array($ext, ['csv', 'xlsx', 'xls'], true)) {
            throw new Exception("Invalid file type. Please upload CSV, XLSX, or XLS.");
        }

        $rows = ($ext === 'csv') ? readCsvRows($tmp) : readExcelRows($tmp);
        if (empty($rows)) throw new Exception("No data found. Ensure your file has a header row and at least one record.");

        $mapped = [];
        foreach ($rows as $row) {
            $policy_number = $row['policy_number'] ?? $row['policy'] ?? $row['policynumber'] ?? '';
            $trans_date    = $row['trans_date'] ?? $row['transaction_date'] ?? $row['date'] ?? '';
            $amount        = $row['amount'] ?? $row['premium'] ?? '';

            $mapped[] = [
                'policy_number' => normalizePolicyNumber((string)$policy_number),
                'trans_date'    => (string)$trans_date,
                'amount'        => $amount,
            ];
        }

        $policySet = fetchPolicySet($con);

        $insertSql = "
            INSERT INTO premium_transactions
            (policy_number, trans_date, amount, userindex, created_by, payment_reference, payment_batch_id)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ";
        $stmtIns = $con->prepare($insertSql);
        if (!$stmtIns) throw new Exception("Prepare failed (insert): " . $con->error);

        $con->begin_transaction();

        $inserted = 0;
        $skipped  = 0;
        $errors   = [];

        foreach ($mapped as $idx => $r) {
            $rowNo = $idx + 2;

            $pn = $r['policy_number'];
            $dt = normalizeDate($r['trans_date']);
            $am = normalizeAmount($r['amount']);

            if ($pn === '' || $dt === null || $am === null) {
                $skipped++;
                $errors[] = "Row {$rowNo}: Missing/invalid fields (policy_number/trans_date/amount).";
                continue;
            }

            if (!isset($policySet[$pn])) {
                $skipped++;
                $errors[] = "Row {$rowNo}: Policy number '{$pn}' not found in policy_holders.";
                continue;
            }

            $am_f = (float)$am;
            $ui   = (int)$userindex;
            $cb   = (int)$created_by;

            $stmtIns->bind_param("ssdiiss", $pn, $dt, $am_f, $ui, $cb, $payment_reference, $payment_batch_id);

            if (!$stmtIns->execute()) {
                $skipped++;
                $errors[] = "Row {$rowNo}: Insert failed (" . $stmtIns->error . ").";
            } else {
                $inserted++;
            }
        }

        $stmtIns->close();
        $con->commit();

        $summary_saved = insertUploadSummary($con, $name, $inserted, $skipped, $payment_batch_id, $payment_reference, $userindex);

        $upload_summary = [
            'file'      => $name,
            'inserted'  => $inserted,
            'skipped'   => $skipped,
            'batch_id'  => $payment_batch_id,
            'reference' => $payment_reference,
            'errors'    => $errors,
            'saved'     => $summary_saved ? 'YES' : 'NO'
        ];

        $success_message = "Upload completed. Inserted: {$inserted}. Skipped: {$skipped}. Batch: {$payment_batch_id}. Summary saved: " . ($summary_saved ? "YES" : "NO");

    } catch (Exception $e) {
        try { $con->rollback(); } catch (Throwable $t) {}
        $error_message = $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Premium Upload</title>
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css"/>

  <style>
    body { font-family: Roboto, Arial, sans-serif; background: #f4f6f9; }
    .content { padding: 20px; }
    .container-fluid { max-width: 1100px; margin: 0 auto; }
    .row { display: flex; flex-wrap: wrap; }
    .justify-content-center { justify-content: center; }
    .col-md-12 { width: 100%; }
    .card { background: #fff; border-radius: 12px; box-shadow: 0 6px 18px rgba(0,0,0,.08); margin-top: 15px; overflow: hidden; }
    .card-header { padding: 25px; }
    .card-header-primary { background: linear-gradient(60deg, #ab47bc, #7b1fa2); }
    .card-title { margin: 0; }
    .card-category { margin: 8px 0 0; }
    .card-body { padding: 20px; }
    label { display:block; font-weight:600; margin-top: 12px; }
    input, select { width: 100%; padding: 11px; border-radius: 10px; border: 1px solid #d9d9d9; margin-top: 6px; }
    .btn { margin-top: 16px; padding: 12px 16px; border-radius: 10px; border: none; cursor: pointer; font-weight: 700; display:inline-block; text-decoration:none; text-align:center; }
    .btn-primary { background: #7b1fa2; color: #fff; }
    .btn-danger { background: #c62828; color: #fff; margin-left:10px; }
    .msg { padding: 12px 14px; border-radius: 10px; margin: 12px 0; }
    .msg-success { background: #e7fff0; border: 1px solid #b7f5ce; color: #0b6b2a; }
    .msg-error { background: #ffe5e5; border: 1px solid #ffb3b3; color: #a10000; }
    .small { color:#666; font-size: 13px; margin-top: 6px; }
    table { width:100%; border-collapse: collapse; margin-top: 10px; }
    th, td { border:1px solid #e6e6e6; padding: 10px; text-align:left; font-size: 13px; }
    th { background:#f5f5f5; }
    ul { margin: 8px 0 0 18px; }
    .actions { margin-top: 10px; }
  </style>
</head>
<body>
    
<?php include 'main_menu_app.php'; ?>
<style>
        body {
    background-image: url('images/ayo_banner4.png'); /* Replace with your actual image path */
    background-size: cover;
  background-position: center;
    background-repeat: no-repeat;
    background-attachment: fixed;
    opacity: 0.9; /* Adjust transparency (0.1 to 1) */
}

body::before {
    content: "";
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-image: url('images/ayo_banner4.png');
    background-size: cover;
    background-position: center;
    background-repeat: no-repeat;
    background-attachment: fixed;
    opacity: 0.5; /* Adjust transparency */
    z-index: -1;
}


</style>

<div class="content">
  <div class="container-fluid">
    <div class="row justify-content-center">
      <div class="col-md-12">

        <div class="card form-card">
          <div class="card-header card-header-primary" style="background: #6ec1e4; color: white; text-align: center; padding: 15px;">
            <h4 class="card-title"style="color:rgb(10, 44, 234); font-weight: bold; text-align: center;">
              <i class="fas fa-upload"></i> PREMIUM UPLOAD
            </h4>
            <p class="card-category" style="color:rgb(10, 44, 234); font-weight: bold; text-align: center;">
              Upload premium transactions (CSV) and validate policy numbers
            </p>
          </div>

          <div class="card-body">

            <?php if ($error_message): ?>
              <div class="msg msg-error"><?php echo htmlspecialchars($error_message); ?></div>
            <?php endif; ?>

            <?php if ($success_message): ?>
              <div class="msg msg-success"><?php echo htmlspecialchars($success_message); ?></div>
            <?php endif; ?>

            <form method="POST" enctype="multipart/form-data" id="uploadForm">
              <label>Upload File (.CSV)</label>
              <input type="file" name="premium_file" accept=".csv,.xlsx,.xls" required>

              <div class="small">
                File must contain headers: <b>policy_number</b>, <b>trans_date</b>, <b>amount</b><br>
                Example header: <code>policy number, date(yyy-mm-dd), premium</code>
                Example row: <code>ABR01260000003, 2026-01-23, 50.00</code>
              </div>

              <label>Payment Reference (optional)</label>
              <input type="text" name="payment_reference" placeholder="e.g. HUBTEL-REF-12345">

              <label>Payment Batch ID (optional)</label>
              <input type="text" name="payment_batch_id" placeholder="Leave empty to auto-generate">

              <div class="actions">
                <button type="submit" name="upload_premiums" class="btn btn-primary">
                  <i class="fas fa-cloud-upload-alt"></i> Upload Premiums
                </button>

                <!-- ✅ CLEAR BUTTON (NEW) -->
                <a class="btn btn-danger" href="?clear=1" onclick="return confirm('Clear the current upload results and start a new upload?');">
                  <i class="fas fa-trash"></i> Clear / New Upload
                </a>
              </div>
            </form>

            <?php if (is_array($upload_summary)): ?>
              <hr style="margin: 20px 0; border:0; border-top:1px solid #eee;">
              <h4 style="margin:0 0 8px 0;">Upload Summary</h4>
              <table>
                <tr><th>File</th><td><?php echo htmlspecialchars($upload_summary['file']); ?></td></tr>
                <tr><th>Inserted</th><td><?php echo (int)$upload_summary['inserted']; ?></td></tr>
                <tr><th>Skipped</th><td><?php echo (int)$upload_summary['skipped']; ?></td></tr>
                <tr><th>Batch ID</th><td><?php echo htmlspecialchars($upload_summary['batch_id']); ?></td></tr>
                <tr><th>Reference</th><td><?php echo htmlspecialchars($upload_summary['reference']); ?></td></tr>
                <tr><th>Summary Saved</th><td><?php echo htmlspecialchars($upload_summary['saved']); ?></td></tr>
              </table>

              <?php if (!empty($upload_summary['errors'])): ?>
                <div class="msg msg-error" style="margin-top: 12px;">
                  <b>Row Issues:</b>
                  <ul>
                    <?php foreach (array_slice($upload_summary['errors'], 0, 20) as $err): ?>
                      <li><?php echo htmlspecialchars($err); ?></li>
                    <?php endforeach; ?>
                  </ul>
                  <?php if (count($upload_summary['errors']) > 20): ?>
                    <div class="small">Showing first 20 errors only.</div>
                  <?php endif; ?>
                </div>
              <?php endif; ?>
            <?php endif; ?>

          </div>
        </div>

      </div>
    </div>
  </div>
</div>

</body>
</html>
