<?php
/**
 * Determine if a policy is lapsed based on premium_transactions and T&Cs.
 *
 * T&C Rules:
 * - Premium cycle: 30 days
 * - Grace period: 30 days
 * - Policy lapses AFTER 60 days from last premium payment
 */
function abraboIsPolicyLapsed(mysqli $con, string $policyNumber, string $asAtDate): array
{
    $out = [
        "ok" => false,
        "is_lapsed" => true,
        "last_paid_date" => null,
        "lapse_date" => null,
        "days_since_last_payment" => null,
        "reason" => ""
    ];

    // Normalize as-at date (date only)
    $asAtNorm = date('Y-m-d', strtotime($asAtDate));
    $asAtTs = strtotime($asAtNorm);
    if ($asAtTs === false) {
        $out["reason"] = "Invalid asAtDate provided.";
        return $out;
    }

    // Get last premium payment on/before asAtDate
    $sql = "
        SELECT MAX(DATE(trans_date)) AS last_paid_date
        FROM premium_transactions
        WHERE policy_number = ?
          AND DATE(trans_date) <= DATE(?)
    ";
    $stmt = $con->prepare($sql);
    if (!$stmt) {
        $out["reason"] = "Prepare failed: " . $con->error;
        return $out;
    }

    $stmt->bind_param("ss", $policyNumber, $asAtNorm);
    $stmt->execute();

    $lastPaid = null;
    $stmt->bind_result($lastPaid);
    $stmt->fetch();
    $stmt->close();

    // No payment at all → lapsed
    if (!$lastPaid) {
        $out["ok"] = true;
        $out["is_lapsed"] = true;
        $out["reason"] = "No premium payment found on/before {$asAtNorm}. Policy not in force.";
        return $out;
    }

    // Normalize last payment date
    $lastPaidNorm = date('Y-m-d', strtotime($lastPaid));
    $lastPaidTs = strtotime($lastPaidNorm);

    // Lapse date = last payment + 60 days
    $lapseTs = strtotime("+60 days", $lastPaidTs);
    $lapseDate = date('Y-m-d', $lapseTs);

    // Days since last payment
    $daysSince = (int) floor(($asAtTs - $lastPaidTs) / 86400);

    // Lapse rule (AFTER 60 days)
    $isLapsed = ($asAtTs > $lapseTs);

    $out["ok"] = true;
    $out["is_lapsed"] = $isLapsed;
    $out["last_paid_date"] = $lastPaidNorm;
    $out["lapse_date"] = $lapseDate;
    $out["days_since_last_payment"] = $daysSince;

    $out["reason"] = $isLapsed
        ? "Policy lapsed: beyond 30-day cycle plus 30-day grace period."
        : "Policy active: within cycle/grace window after last payment.";

    return $out;
}
