<?php
/**
 * Process a successful payment callback:
 * - Update subscription_main payment_status
 * - Read subscription_main by policy_number
 * - Insert customer if not exists (by mobile_money_number)
 * - Insert policy_holders (if not already created)
 * - Copy subscription_additionallife -> secondary_lives
 * - Copy subscription_beneficiaries -> beneficiaries
 *
 * Usage:
 *   include("db_connection/database.php");
 *   require_once("functions/process_policy_success.php");
 *   $result = processPolicySuccess($con, $policyNumberFromApi, 'Success');
 *   echo $result['message'];
 */

function processPolicySuccess(mysqli $con, string $policyNumber, string $responseStatus = 'Success'): array
{
    // Enable mysqli exceptions (optional, but recommended)
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

    $policyNumber = trim($policyNumber);
    $status_input_date = date("Y-m-d H:i:s");
    $policy_status = 'Active';

    try {
        $con->begin_transaction();

        // =====================================================
        // 1) Update subscription_main payment_status
        // =====================================================
        $stmtUpdate = $con->prepare("
            UPDATE subscription_main
            SET payment_status = ?,
                status_input_date = ?
            WHERE policy_number = ?
        ");
        $stmtUpdate->bind_param("sss", $responseStatus, $status_input_date, $policyNumber);
        $stmtUpdate->execute();
        $stmtUpdate->close();

        // =====================================================
        // 2) Fetch subscription_main by policy_number (NO TRIM to allow index)
        // =====================================================
        $stmtSubMain = $con->prepare("
            SELECT
                full_name,
                date_of_birth,
                gender,
                mobile_money_number,
                userindex,
                policy_number,
                product_name,
                total_premium,
                onboarding_channel,
                mandate_id,
                status_input_date,
                momo_network,
                subscriptions_id
            FROM subscription_main
            WHERE policy_number = ?
            LIMIT 1
        ");
        $stmtSubMain->bind_param("s", $policyNumber);
        $stmtSubMain->execute();
        $subRes = $stmtSubMain->get_result();
        $sub = $subRes->fetch_assoc();
        $stmtSubMain->close();

        if (!$sub) {
            throw new Exception("subscription_main not found for policy_number: {$policyNumber}");
        }

        // Customer fields
        $full_name           = $sub['full_name'];
        $date_of_birth       = $sub['date_of_birth'];
        $gender              = $sub['gender'];
        $mobile_money_number = trim($sub['mobile_money_number']);
        $userindex           = (int)$sub['userindex'];

        // Policy fields
        $policy_number      = $sub['policy_number'];
        $product_name       = $sub['product_name'];
        $total_premium      = (float)$sub['total_premium'];
        $onboarding_channel = $sub['onboarding_channel'];
        $mandate_id         = $sub['mandate_id'];
        $inception_date     = $sub['status_input_date']; // or use $status_input_date if you prefer
        $momo_network       = $sub['momo_network'];
        $subscriptions_id   = (int)$sub['subscriptions_id'];

        // =====================================================
        // 3) Customer: get or insert
        // =====================================================
        $stmtChkCust = $con->prepare("
            SELECT cust_id
            FROM customer
            WHERE mobile_money_number = ?
            LIMIT 1
        ");
        $stmtChkCust->bind_param("s", $mobile_money_number);
        $stmtChkCust->execute();
        $custRow = $stmtChkCust->get_result()->fetch_assoc();
        $stmtChkCust->close();

        if ($custRow) {
            $cust_id = (int)$custRow['cust_id'];
        } else {
            $stmtInsCust = $con->prepare("
                INSERT INTO customer
                (full_name, date_of_birth, gender, mobile_money_number, userindex)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmtInsCust->bind_param("ssssi", $full_name, $date_of_birth, $gender, $mobile_money_number, $userindex);
            $stmtInsCust->execute();
            $cust_id = (int)$con->insert_id;
            $stmtInsCust->close();
        }

        // =====================================================
        // 4) Get product_index
        // =====================================================
        $stmtProd = $con->prepare("
            SELECT product_index
            FROM product
            WHERE product_name = ?
            LIMIT 1
        ");
        $stmtProd->bind_param("s", $product_name);
        $stmtProd->execute();
        $prodRow = $stmtProd->get_result()->fetch_assoc();
        $stmtProd->close();

        if (!$prodRow) {
            throw new Exception("Product not found for product_name: {$product_name}");
        }
        $product_index = (int)$prodRow['product_index'];

        // =====================================================
        // 5) Policy holder: get or insert
        // =====================================================
        $stmtChkPol = $con->prepare("
            SELECT policyholder_id
            FROM policy_holders
            WHERE subscriptions_id = ?
            LIMIT 1
        ");
        $stmtChkPol->bind_param("s", $subscriptions_id);
        $stmtChkPol->execute();
        $polRow = $stmtChkPol->get_result()->fetch_assoc();
        $stmtChkPol->close();
        if ($polRow) {
            $policyholder_id = (int)$polRow['policyholder_id'];
        } else {
            $stmtInsPol = $con->prepare("
                INSERT INTO policy_holders
                (
                    cust_id,
                    momo_network,
                    product_index,
                    set_premium,
                    mandate_id,
                    onboarding_channel,
                    policy_status,
                    inception_date,
                    userindex,
                    subscriptions_id
                )
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            // product_index is int => i
            $stmtInsPol->bind_param(
                "isidssssii",
                $cust_id,
                $momo_network,
                $product_index,
                $total_premium,
                $mandate_id,
                $onboarding_channel,
                $policy_status,
                $inception_date,
                $userindex,
                $subscriptions_id
            );

            $stmtInsPol->execute();
            $policyholder_id = (int)$con->insert_id;
            $stmtInsPol->close();
        }

        // =====================================================
        // 6) Copy subscription_additionallife -> secondary_lives
        // (prepare INSERT once for speed)
        // =====================================================
        $stmtAddLifeSel = $con->prepare("
            SELECT full_name, relationship, date_of_birth,mobile_money_number,gender
            FROM subscription_additionallife
            WHERE subscriptions_id = ?
        ");
        $stmtAddLifeSel->bind_param("i", $subscriptions_id);
        $stmtAddLifeSel->execute();
        $rsAddLife = $stmtAddLifeSel->get_result();

        $stmtSecIns = $con->prepare("
            INSERT INTO secondary_lives
            (
                policyholder_id,
                full_name,
                relationship,
                date_of_birth,
                age_at_entry,
                created_at,
                userindex,
                mobile_money_number,
                gender,
                subscriptions_id
            )
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        while ($row = $rsAddLife->fetch_assoc()) {
            $sec_full_name    = $row['full_name'];
            $sec_relationship = $row['relationship'];
            $sec_dob          = $row['date_of_birth'];
            $sec_mobile_money_number = $row['mobile_money_number'];
            $sec_gender = $row['gender'];
            
            

            $dobObj = new DateTime($sec_dob);
            $age_at_entry = (new DateTime())->diff($dobObj)->y;
            $created_at = date("Y-m-d H:i:s");

            // Optional: duplicate prevention
            $stmtSecDup = $con->prepare("
                SELECT secondary_life_id
                FROM secondary_lives
                WHERE policyholder_id = ?
                  AND full_name = ?
                  AND date_of_birth = ?
                LIMIT 1
            ");
            $stmtSecDup->bind_param("iss", $policyholder_id, $sec_full_name, $sec_dob);
            $stmtSecDup->execute();
            $dup = $stmtSecDup->get_result()->fetch_assoc();
            $stmtSecDup->close();

            if ($dup) continue;

            $stmtSecIns->bind_param(
                "isssisisss",
                $policyholder_id,
                $sec_full_name,
                $sec_relationship,
                $sec_dob,
                $age_at_entry,
                $created_at,
                $userindex,
                $sec_mobile_money_number,
                $sec_gender,
                $subscriptions_id
            );
            $stmtSecIns->execute();
        }

        $stmtSecIns->close();
        $stmtAddLifeSel->close();

        // =====================================================
        // 7) Copy subscription_beneficiaries -> beneficiaries
        // (prepare INSERT once for speed)
        // =====================================================
        $stmtBenSel = $con->prepare("
            SELECT full_name, relationship, mobile_money_number
            FROM subscription_beneficiaries
            WHERE subscriptions_id = ?
        ");
        $stmtBenSel->bind_param("i", $subscriptions_id);
        $stmtBenSel->execute();
        $rsBen = $stmtBenSel->get_result();

        $stmtBenIns = $con->prepare("
            INSERT INTO beneficiaries
            (
                policyholder_id,
                full_name,
                relationship,
                mobile_money_number,
                created_at,
                subscriptions_id
            )
            VALUES (?, ?, ?, ?, ?, ?)
        ");

        while ($row = $rsBen->fetch_assoc()) {
            $ben_full_name = $row['full_name'];
            $ben_relation  = $row['relationship'];
            $ben_mobile    = $row['mobile_money_number'];
            $created_at    = date("Y-m-d H:i:s");

            // Optional: duplicate prevention (change beneficiary_id if your PK name differs)
            $stmtBenDup = $con->prepare("
                SELECT beneficiary_id
                FROM beneficiaries
                WHERE policyholder_id = ?
                  AND full_name = ?
                  AND relationship = ?
                  AND mobile_money_number = ?
                LIMIT 1
            ");
            $stmtBenDup->bind_param("isss", $policyholder_id, $ben_full_name, $ben_relation, $ben_mobile);
            $stmtBenDup->execute();
            $dupB = $stmtBenDup->get_result()->fetch_assoc();
            $stmtBenDup->close();

            if ($dupB) continue;

            $stmtBenIns->bind_param("isssss", $policyholder_id, $ben_full_name, $ben_relation, $ben_mobile, $created_at,$subscriptions_id);
            $stmtBenIns->execute();
        }

        $stmtBenIns->close();
        $stmtBenSel->close();

        // Commit
        $con->commit();

        return [
            'success' => true,
            'policy_number' => $policyNumber,
            'message' => "DONE: processed policy {$policyNumber} successfully."
        ];

    } catch (Throwable $e) {
        $con->rollback();
        return [
            'success' => false,
            'policy_number' => $policyNumber,
            'message' => $e->getMessage()
        ];
    }
}
