<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include("database.php");
include("session.php");

date_default_timezone_set("Africa/Accra");

// Session
$userindex     = isset($_SESSION['userindex']) ? (int)$_SESSION['userindex'] : 0;
$user_sub_priv = isset($_SESSION['user_sub_priv']) ? trim($_SESSION['user_sub_priv']) : '';

// Inputs (POST/GET)
$date_from      = isset($_REQUEST['date_from']) ? trim($_REQUEST['date_from']) : '';
$date_to        = isset($_REQUEST['date_to']) ? trim($_REQUEST['date_to']) : '';
$payment_status = isset($_REQUEST['payment_status']) ? trim($_REQUEST['payment_status']) : 'ALL';
$selected_user  = isset($_REQUEST['selected_user']) ? trim($_REQUEST['selected_user']) : 'ALL';

// Defaults
if ($date_from === '') $date_from = date('Y-m-01');
if ($date_to === '')   $date_to   = date('Y-m-d');

// Normalize date
$date_from = date("Y-m-d", strtotime($date_from));
$date_to   = date("Y-m-d", strtotime($date_to));

// Basic validation
if ($date_from > $date_to) {
    die("Error: Date From cannot be greater than Date To.");
}

/**
 * Build SQL using prepared statements (clean + safe)
 * View fields assumed: input_date, userindex, payment_status, etc.
 */
$sql = "SELECT *
        FROM view_telesales_report
        WHERE DATE(input_date) BETWEEN ? AND ?";

$params = [$date_from, $date_to];
$types  = "ss";

// Privilege filters
if ($user_sub_priv === '106') {
    // Agent sees own records only
    $sql .= " AND userindex = ?";
    $params[] = $userindex;
    $types .= "i";
} elseif ($user_sub_priv === '107') {
    // Admin can filter user
    if ($selected_user !== 'ALL' && $selected_user !== '') {
        $sql .= " AND userindex = ?";
        $params[] = (int)$selected_user;
        $types .= "i";
    }
}

// Payment status filter
if ($payment_status !== 'ALL' && $payment_status !== '') {
    $sql .= " AND payment_status = ?";
    $params[] = $payment_status;
    $types .= "s";
}

$sql .= " ORDER BY input_date DESC";

// Export handlers
$export = isset($_GET['export']) ? $_GET['export'] : '';
if ($export === 'excel') {
    // Export to Excel (TSV)
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="subscriptions_report_'.$date_from.'_to_'.$date_to.'.xls"');

    $stmt = $con->prepare($sql);
    if (!$stmt) die("Prepare failed: " . $con->error);

    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $res = $stmt->get_result();

    echo "No\tDate\tMobile No.\tName\tDoB\tGender\tPolicy No.\tProduct\tPremium\tStatus\tChannel\tUser\n";

    $no = 0;
    while ($row = $res->fetch_assoc()) {
        $no++;

        $input_date = !empty($row['input_date']) ? date("d-m-Y", strtotime($row['input_date'])) : '';
        $dob        = !empty($row['date_of_birth']) ? date("d-m-Y", strtotime($row['date_of_birth'])) : '';
        $premium    = number_format((float)($row['total_premium'] ?? 0), 2);
        $tuser      = trim(($row['user_surname'] ?? '') . ' ' . ($row['user_othernames'] ?? ''));

        echo $no . "\t"
           . ($input_date) . "\t"
           . ($row['mobile_money_number'] ?? '') . "\t"
           . ($row['full_name'] ?? '') . "\t"
           . ($dob) . "\t"
           . ($row['gender'] ?? '') . "\t"
           . ($row['policy_number'] ?? '') . "\t"
           . ($row['product_name'] ?? '') . "\t"
           . ($premium) . "\t"
           . ($row['payment_status'] ?? '') . "\t"
           . ($row['onboarding_channel'] ?? '') . "\t"
           . ($tuser) . "\n";
    }

    $stmt->close();
    exit;
}

if ($export === 'pdf') {
    // Export to PDF using TCPDF
    require_once('tcpdf/tcpdf.php');

    $stmt = $con->prepare($sql);
    if (!$stmt) die("Prepare failed: " . $con->error);

    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $res = $stmt->get_result();

    $pdf = new TCPDF('L', 'mm', 'A4', true, 'UTF-8', false);
    $pdf->SetCreator('Report System');
    $pdf->SetAuthor('ImpPAS');
    $pdf->SetTitle('Subscriptions Report');
    $pdf->SetHeaderData('', 0, 'Subscriptions Report', "From: $date_from  To: $date_to");
    $pdf->setHeaderFont(['helvetica', '', 10]);
    $pdf->setFooterFont(['helvetica', '', 8]);
    $pdf->SetMargins(10, 25, 10);
    $pdf->SetHeaderMargin(5);
    $pdf->SetFooterMargin(10);
    $pdf->SetAutoPageBreak(true, 12);
    $pdf->SetFont('helvetica', '', 8);
    $pdf->AddPage();

    $html = '<h3 style="text-align:center;color:#2a89c9;">Subscriptions Report</h3>';
    $html .= '<table border="1" cellpadding="3" style="border-collapse:collapse;font-size:8px;">';
    $html .= '<thead>
        <tr style="background-color:#2a89c9;color:#fff;">
          <th width="30">No</th>
          <th width="60">Date</th>
          <th width="85">Mobile</th>
          <th width="140">Name</th>
          <th width="60">DoB</th>
          <th width="45">Gender</th>
          <th width="90">Policy No</th>
          <th width="120">Product</th>
          <th width="70">Premium</th>
          <th width="60">Status</th>
          <th width="80">Channel</th>
          <th width="110">User</th>
        </tr>
      </thead><tbody>';

    $no = 0;
    if ($res->num_rows > 0) {
        while ($row = $res->fetch_assoc()) {
            $no++;
            $input_date = !empty($row['input_date']) ? date("d-m-Y", strtotime($row['input_date'])) : '';
            $dob        = !empty($row['date_of_birth']) ? date("d-m-Y", strtotime($row['date_of_birth'])) : '';
            $premium    = number_format((float)($row['total_premium'] ?? 0), 2);
            $tuser      = trim(($row['user_surname'] ?? '') . ' ' . ($row['user_othernames'] ?? ''));

            $html .= '<tr>
              <td width="30">'.$no.'</td>
              <td width="60">'.htmlspecialchars($input_date).'</td>
              <td width="85">'.htmlspecialchars($row['mobile_money_number'] ?? '').'</td>
              <td width="140">'.htmlspecialchars($row['full_name'] ?? '').'</td>
              <td width="60">'.htmlspecialchars($dob).'</td>
              <td width="45">'.htmlspecialchars($row['gender'] ?? '').'</td>
              <td width="90">'.htmlspecialchars($row['policy_number'] ?? '').'</td>
              <td width="120">'.htmlspecialchars($row['product_name'] ?? '').'</td>
              <td width="70" align="right">'.htmlspecialchars($premium).'</td>
              <td width="60">'.htmlspecialchars($row['payment_status'] ?? '').'</td>
              <td width="80">'.htmlspecialchars($row['onboarding_channel'] ?? '').'</td>
              <td width="110">'.htmlspecialchars($tuser).'</td>
            </tr>';
        }
    } else {
        $html .= '<tr><td colspan="12" align="center">No records found</td></tr>';
    }

    $html .= '</tbody></table>';

    $pdf->writeHTML($html, true, false, true, false, '');
    $pdf->Output('subscriptions_report_'.$date_from.'_to_'.$date_to.'.pdf', 'D');

    $stmt->close();
    exit;
}

// Normal page display (no export)
$stmt = $con->prepare($sql);
if (!$stmt) {
    die("Prepare failed: " . $con->error);
}
$stmt->bind_param($types, ...$params);
$stmt->execute();
$resultuser_user = $stmt->get_result();

$no = 0;

// Keep current filters for export links
$qs = http_build_query([
    'date_from' => $date_from,
    'date_to' => $date_to,
    'payment_status' => $payment_status,
    'selected_user' => $selected_user
]);
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <link rel="apple-touch-icon" sizes="76x76" href="../assets/img/apple-icon.png">
  <link rel="icon" type="image/png" href="../assets/img/favicon.png">
  <meta http-equiv="X-UA-Compatible" content="IE=edge,chrome=1" />
  <title>Subscriptions Report</title>
  <meta content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=0, shrink-to-fit=no" name="viewport" />
  <link href="../assets/css/material-dashboard.css?v=2.1.1" rel="stylesheet" />
  <link href="../assets/demo/demo.css" rel="stylesheet" />
  
  <style>
    body {
      font-family: Arial, sans-serif;
      margin: 0;
      color: #333;
      background: url('images/ayo_banner4.png') center/cover no-repeat fixed;
    }
    body::before {
      content: "";
      position: fixed;
      inset: 0;
      background: url('images/ayo_banner4.png') center/cover no-repeat fixed;
      opacity: 0.35;
      z-index: -1;
    }
    .card {
      border-radius: 12px;
      box-shadow: 0 6px 18px rgba(0,0,0,0.12);
      margin-bottom: 20px;
    }
    .card-header-primary {
      background: linear-gradient(60deg, #6ec1e4, #2a89c9) !important;
      color: #fff;
      text-align: center;
      padding: 20px 15px;
      border-radius: 12px 12px 0 0;
    }
    .card-header-primary h4 {
      margin: 0;
      font-weight: bold;
      font-size: 1.4rem;
    }
    .card-header-primary p {
      margin: 8px 0 0;
      font-size: 0.95rem;
      opacity: 0.9;
    }
    .header-actions {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-top: 15px;
    }
    .export-buttons {
      display: flex;
      gap: 10px;
    }
    .export-buttons a {
      font-weight: bold;
      border-radius: 5px;
      padding: 8px 20px;
      display: flex;
      align-items: center;
      gap: 5px;
    }
    .back-button {
      background-color: #6c757d;
      color: white;
      border: none;
      border-radius: 5px;
      padding: 8px 20px;
      font-weight: bold;
      cursor: pointer;
      text-decoration: none;
      display: flex;
      align-items: center;
      gap: 5px;
    }
    .back-button:hover {
      background-color: #5a6268;
      color: white;
      text-decoration: none;
    }
    .table-responsive {
      border-radius: 8px;
      overflow: hidden;
      max-height: 70vh;
      overflow-y: auto;
    }
    #report_data {
      margin-bottom: 0;
      width: 100%;
      table-layout: fixed;
    }
    #report_data thead {
      position: sticky;
      top: 0;
      z-index: 10;
    }
    #report_data thead th {
      background-color: #2a89c9;
      color: white;
      font-weight: bold;
      text-align: center;
      padding: 12px 8px;
      border: none;
      position: relative;
    }
    #report_data tbody td {
      padding: 10px 8px;
      vertical-align: middle;
      border: 1px solid #dee2e6;
      word-wrap: break-word;
      overflow-wrap: break-word;
    }
    #report_data tbody tr:nth-child(even) {
      background-color: #f8f9fa;
    }
    #report_data tbody tr:hover {
      background-color: #e9f7fe;
      transition: background-color 0.2s;
    }
    .text-truncate {
      white-space: nowrap;
      overflow: hidden;
      text-overflow: ellipsis;
    }
    .no-data {
      text-align: center;
      padding: 40px !important;
      font-size: 1.1rem;
      color: #6c757d;
    }
    /* Column specific widths */
    .col-no { width: 50px; min-width: 50px; }
    .col-date { width: 90px; min-width: 90px; }
    .col-mobile { width: 110px; min-width: 110px; }
    .col-name { width: 180px; min-width: 180px; }
    .col-dob { width: 90px; min-width: 90px; }
    .col-gender { width: 70px; min-width: 70px; }
    .col-policy { width: 130px; min-width: 130px; }
    .col-product { width: 150px; min-width: 150px; }
    .col-premium { width: 100px; min-width: 100px; }
    .col-status { width: 90px; min-width: 90px; }
    .col-channel { width: 110px; min-width: 110px; }
    .col-user { width: 150px; min-width: 150px; }
    
    /* For smaller screens */
    @media (max-width: 768px) {
      .table-responsive {
        font-size: 0.9rem;
      }
      .col-name, .col-product, .col-user {
        width: 120px;
        min-width: 120px;
      }
    }
  </style>
</head>

<body>
<?php include 'main_menu_app.php' ?>

<div class="content">
  <div class="container-fluid">
    <div class="row">
      <div class="col-md-12">
        <div class="card">

          <div class="card-header card-header-primary">
            <h4>Subscriptions Report</h4>
            <p>
              From <b><?php echo htmlspecialchars($date_from); ?></b> to <b><?php echo htmlspecialchars($date_to); ?></b>
              <?php if ($payment_status !== 'ALL') : ?> | Status: <b><?php echo htmlspecialchars($payment_status); ?></b><?php endif; ?>
              <?php if ($selected_user !== 'ALL' && $selected_user !== '') : ?> | User ID: <b><?php echo htmlspecialchars($selected_user); ?></b><?php endif; ?>
            </p>

            <div class="header-actions">
              <a href="javascript:history.back()" class="back-button">
                <i class="material-icons">arrow_back</i> Go Back
              </a>
              
              <div class="export-buttons">
                <a class="btn btn-success" href="?export=excel&<?php echo $qs; ?>">
                  <i class="material-icons">description</i> Excel
                </a>
                <a class="btn btn-danger" href="?export=pdf&<?php echo $qs; ?>">
                  <i class="material-icons">picture_as_pdf</i> PDF
                </a>
              </div>
            </div>
          </div>

          <div class="card-body">
            <div class="table-responsive">
              <table class="table table-bordered table-striped" id="report_data">
                <thead>
                  <tr>
                    <th class="col-no text-center">No.</th>
                    <th class="col-date text-center">Date</th>
                    <th class="col-mobile">Mobile No.</th>
                    <th class="col-name">Name</th>
                    <th class="col-dob text-center">DoB</th>
                    <th class="col-gender text-center">Gender</th>
                    <th class="col-policy">Policy No.</th>
                    <th class="col-product">Product</th>
                    <th class="col-premium text-right">Premium</th>
                    <th class="col-status text-center">Status</th>
                    <th class="col-channel text-center">Channel</th>
                    <th class="col-user">User</th>
                  </tr>
                </thead>
                <tbody>
                <?php if ($resultuser_user && $resultuser_user->num_rows > 0) : ?>
                  <?php while ($test_user = $resultuser_user->fetch_assoc()) :
                      $no++;

                      $input_date = !empty($test_user['input_date']) ? date("d-m-Y", strtotime($test_user['input_date'])) : '';
                      $dob        = !empty($test_user['date_of_birth']) ? date("d-m-Y", strtotime($test_user['date_of_birth'])) : '';
                      $premium    = number_format((float)($test_user['total_premium'] ?? 0), 2);
                      $tuser      = trim(($test_user['user_surname'] ?? '') . ' ' . ($test_user['user_othernames'] ?? ''));
                      
                      // Add status color coding
                      $status_class = '';
                      if (isset($test_user['payment_status'])) {
                          switch(strtoupper($test_user['payment_status'])) {                              
                              case 'SUCCESS':
                                  $status_class = 'text-success font-weight-bold';
                                  break;
                              case 'PENDING':
                                  $status_class = 'text-warning font-weight-bold';
                                  break;
                              case 'FAILED':                              
                                  $status_class = 'text-danger font-weight-bold';
                                  break;
                          }
                      }
                  ?>
                    <tr>
                      <td class="col-no text-center"><?php echo $no; ?></td>
                      <td class="col-date text-center"><?php echo htmlspecialchars($input_date); ?></td>
                      <td class="col-mobile text-truncate" title="<?php echo htmlspecialchars($test_user['mobile_money_number'] ?? ''); ?>">
                        <?php echo htmlspecialchars($test_user['mobile_money_number'] ?? ''); ?>
                      </td>
                      <td class="col-name text-truncate" title="<?php echo htmlspecialchars($test_user['full_name'] ?? ''); ?>">
                        <?php echo htmlspecialchars($test_user['full_name'] ?? ''); ?>
                      </td>
                      <td class="col-dob text-center"><?php echo htmlspecialchars($dob); ?></td>
                      <td class="col-gender text-center"><?php echo htmlspecialchars($test_user['gender'] ?? ''); ?></td>
                      <td class="col-policy text-truncate" title="<?php echo htmlspecialchars($test_user['policy_number'] ?? ''); ?>">
                        <?php echo htmlspecialchars($test_user['policy_number'] ?? ''); ?>
                      </td>
                      <td class="col-product text-truncate" title="<?php echo htmlspecialchars($test_user['product_name'] ?? ''); ?>">
                        <?php echo htmlspecialchars($test_user['product_name'] ?? ''); ?>
                      </td>
                      <td class="col-premium text-right"><?php echo htmlspecialchars($premium); ?></td>
                      <td class="col-status text-center <?php echo $status_class; ?>">
                        <?php echo htmlspecialchars($test_user['payment_status'] ?? ''); ?>
                      </td>
                      <td class="col-channel text-center"><?php echo htmlspecialchars($test_user['onboarding_channel'] ?? ''); ?></td>
                      <td class="col-user text-truncate" title="<?php echo htmlspecialchars($tuser); ?>">
                        <?php echo htmlspecialchars($tuser); ?>
                      </td>
                    </tr>
                  <?php endwhile; ?>
                <?php else : ?>
                  <tr>
                    <td colspan="12" class="no-data">
                      <i class="material-icons" style="font-size: 48px; color: #6c757d;">search_off</i>
                      <div style="margin-top: 10px;">No records found for the selected filters.</div>
                      <a href="javascript:history.back()" class="btn btn-info mt-3">
                        <i class="material-icons">arrow_back</i> Go Back to Search
                      </a>
                    </td>
                  </tr>
                <?php endif; ?>
                </tbody>
              </table>
            </div>
            
            <?php if ($resultuser_user && $resultuser_user->num_rows > 0) : ?>
            <div class="row mt-3">
              <div class="col-md-6">
                <div class="alert alert-info" style="margin: 0;">
                  <i class="material-icons">info</i>
                  <strong>Total Records:</strong> <?php echo $no; ?> record(s) found
                </div>
              </div>
              <div class="col-md-6 text-right">
                <a href="javascript:history.back()" class="btn btn-secondary">
                  <i class="material-icons">arrow_back</i> Go Back
                </a>
              </div>
            </div>
            <?php endif; ?>
          </div>

        </div>
      </div>
    </div>
  </div>
</div>

<?php
$stmt->close();
$con->close();
?>
</body>
</html>