<?php
// ================================
// ImpPAS Dashboard (Fixed Version)
// ================================

// TEMP: Enable error reporting (disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Connect to Database
include("database.php");
include("session.php");

// Get user index (assume numeric)
$full_namen = isset($_SESSION['userindex']) ? (int)$_SESSION['full_name'] : 0;


// Basic guards
if (!isset($con) || !$con) {
    die("Database connection variable \$con not found. Check database.php");
}
if ($con->connect_error) {
    die("DB connection failed: " . $con->connect_error);
}

// Helpers
function safe_session($key, $default='') {
    return isset($_SESSION[$key]) ? (string)$_SESSION[$key] : $default;
}
function safe_req($key, $default='') {
    return isset($_REQUEST[$key]) ? (string)$_REQUEST[$key] : $default;
}
function qval($con, $sql, $default = 0) {
    $res = $con->query($sql);
    if (!$res) {
        die("<pre>SQL Error:\n$sql\n\n".$con->error."</pre>");
    }
    $row = $res->fetch_assoc();
    $res->close();
    if (!$row) return $default;
    $firstKey = array_key_first($row);
    return $row[$firstKey] ?? $default;
}

$userindex     = safe_session('userindex');
$msg           = safe_req('msg');
$user_sub_priv = safe_session('user_sub_priv');

// ======================================
// Fetch actual data from database (FIXED)
// ======================================

// 1. Total Policies
$total_policies = (int) qval($con,
    "SELECT COUNT(*) as total FROM policy_details_view WHERE policy_status = 'ACTIVE'",
    0
);

// 2. Monthly Premiums
$monthly_premiums = (float) qval($con,
    "SELECT COALESCE(SUM(set_premium),0) as total_premium FROM policy_details_view WHERE policy_status = 'ACTIVE'",
    0
);

// 3. Active Claims
$active_claims = (int) qval($con,
    "SELECT COUNT(*) as total FROM claims WHERE claim_status = 'SUBMITTED'",
    0
);

// 4. Pending Approvals
$pending_approvals = (int) qval($con,
    "SELECT COUNT(*) as total FROM policy_details_view WHERE policy_status = 'PENDING'",
    0
);

// Detect correct date column in policy_details_view (created_at often missing in views)
$dateCol = null;
$colCheck = $con->query("SHOW COLUMNS FROM policy_details_view");
if (!$colCheck) {
    die("<pre>SQL Error:\nSHOW COLUMNS FROM policy_details_view\n\n".$con->error."</pre>");
}
$cols = [];
while ($c = $colCheck->fetch_assoc()) $cols[] = $c['Field'];
$colCheck->close();

if (in_array('created_at', $cols)) $dateCol = 'created_at';
elseif (in_array('updated_at', $cols)) $dateCol = 'updated_at';
elseif (in_array('inception_date', $cols)) $dateCol = 'inception_date';
else $dateCol = null;

// 5. Recent Policy Activity (fixed)
if ($dateCol) {
    $recent_activity_query = "SELECT 
        policy_number,
        full_name,
        product_name,
        set_premium,
        policy_status,
        DATE($dateCol) as created_date
        FROM policy_details_view 
        ORDER BY $dateCol DESC 
        LIMIT 5";
} else {
    // Fallback: if no date column exists in the view
    $recent_activity_query = "SELECT 
        policy_number,
        full_name,
        product_name,
        set_premium,
        policy_status,
        '' as created_date
        FROM policy_details_view 
        ORDER BY policyholder_id DESC 
        LIMIT 5";
}

$recent_activity_result = $con->query($recent_activity_query);
if (!$recent_activity_result) {
    die("<pre>SQL Error:\n$recent_activity_query\n\n".$con->error."</pre>");
}

// 6. Premium Collection Data (fixed for DATE or text dates)
$transDateType = 'text';
$typeRes = $con->query("SHOW COLUMNS FROM premium_transactions LIKE 'trans_date'");
if ($typeRes && $typeRow = $typeRes->fetch_assoc()) {
    $type = strtolower($typeRow['Type']);
    if (strpos($type, 'date') !== false || strpos($type, 'time') !== false) {
        $transDateType = 'date';
    }
}
if ($typeRes) $typeRes->close();

if ($transDateType === 'date') {
    $premium_collection_query = "SELECT 
        trans_date,
        amount
        FROM premium_transactions 
        WHERE trans_date IS NOT NULL
        ORDER BY trans_date";
} else {
    $premium_collection_query = "SELECT 
        trans_date,
        amount
        FROM premium_transactions 
        WHERE trans_date IS NOT NULL AND TRIM(trans_date) <> ''
        ORDER BY STR_TO_DATE(trans_date, '%m/%d/%Y')";
}

$premium_collection_result = $con->query($premium_collection_query);
if (!$premium_collection_result) {
    die("<pre>SQL Error:\n$premium_collection_query\n\n".$con->error."</pre>");
}

$premium_data_by_month = [];
$premium_labels = [];
$premium_data = [];

if ($premium_collection_result && $premium_collection_result->num_rows > 0) {
    while ($row = $premium_collection_result->fetch_assoc()) {

        $amount = (float)($row['amount'] ?? 0);

        if ($transDateType === 'date') {
            $ts = strtotime($row['trans_date']);
            if (!$ts) continue;
            $month_num = (int)date('n', $ts);
            $year = (int)date('Y', $ts);
        } else {
            $date_parts = explode('/', (string)$row['trans_date']);
            if (count($date_parts) != 3) continue;
            $month_num = (int)$date_parts[0];
            $year = (int)$date_parts[2];
        }

        $month_names = ['', 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
        $month_name = $month_names[$month_num] ?? 'Unknown';

        $key = $year . '-' . str_pad($month_num, 2, '0', STR_PAD_LEFT);

        if (!isset($premium_data_by_month[$key])) {
            $premium_data_by_month[$key] = [
                'month' => $month_name,
                'year'  => $year,
                'total' => 0
            ];
        }
        $premium_data_by_month[$key]['total'] += $amount;
    }

    ksort($premium_data_by_month);

    // Get last 6 months
    $last_six_months = array_slice($premium_data_by_month, -6, 6, true);

    foreach ($last_six_months as $data) {
        $premium_labels[] = $data['month'];
        $premium_data[]   = $data['total'];
    }
} else {
    // If no data, show zeros (avoid fake demo numbers)
    $premium_labels = ['Sep', 'Oct', 'Nov', 'Dec', 'Jan', 'Feb'];
    $premium_data   = [0, 0, 0, 0, 0, 0];
}

// 7. Policy Distribution by Product
$policy_distribution_query = "SELECT 
    product_name,
    COUNT(*) as count
    FROM policy_details_view 
    WHERE policy_status = 'ACTIVE'
    GROUP BY product_name";

$policy_distribution_result = $con->query($policy_distribution_query);
if (!$policy_distribution_result) {
    die("<pre>SQL Error:\n$policy_distribution_query\n\n".$con->error."</pre>");
}

$policy_labels = [];
$policy_data = [];

if ($policy_distribution_result && $policy_distribution_result->num_rows > 0) {
    while ($row = $policy_distribution_result->fetch_assoc()) {
        $policy_labels[] = $row['product_name'];
        $policy_data[]   = (int)$row['count'];
    }
} else {
    $policy_labels = ['No Data'];
    $policy_data   = [1];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Abrabo Pa Dashboard</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        :root {
            --primary: #2563eb;
            --primary-dark: #1d4ed8;
            --primary-light: #dbeafe;
            --secondary: #10b981;
            --accent: #f59e0b;
            --danger: #ef4444;
            --warning: #f59e0b;
            --gray-50: #f9fafb;
            --gray-100: #f3f4f6;
            --gray-200: #e5e7eb;
            --gray-300: #d1d5db;
            --gray-400: #9ca3af;
            --gray-500: #6b7280;
            --gray-600: #4b5563;
            --gray-700: #374151;
            --gray-800: #1f2937;
            --gray-900: #111827;
            --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
            --shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
            --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
            --radius: 8px;
            --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }

        * { box-sizing: border-box; margin: 0; padding: 0; }

        body {
            font-family: 'Roboto', sans-serif;
            background: linear-gradient(135deg, #f0f4f8 0%, #d9e2ec 100%);
            color: var(--gray-800);
            line-height: 1.6;
            min-height: 100vh;
            /* IMPORTANT FIX: do NOT set opacity on body (it makes everything fade) */
        }

        .dashboard-container {
            display: grid;
            grid-template-columns: 250px 1fr;
            min-height: 100vh;
        }

        /* Sidebar Styles */
        .sidebar {
            background: linear-gradient(180deg, var(--gray-800) 0%, var(--gray-900) 100%);
            color: white;
            padding: 20px 0;
            box-shadow: var(--shadow-lg);
            z-index: 10;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 0 20px 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
            margin-bottom: 20px;
        }

        .logo h1 { font-size: 20px; font-weight: 600; }
        .logo .material-icons { font-size: 28px; color: var(--primary); }

        .nav-menu { list-style: none; padding: 0 10px; }
        .nav-item { margin-bottom: 8px; }

        .nav-link {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 12px 16px;
            color: var(--gray-300);
            text-decoration: none;
            border-radius: var(--radius);
            transition: var(--transition);
        }

        .nav-link:hover, .nav-link.active {
            background: rgba(255, 255, 255, 0.1);
            color: white;
        }

        .nav-link.active { background: var(--primary); }
        .nav-link .material-icons { font-size: 20px; }

        /* Main Content Styles */
        .main-content {
            padding: 24px;
            overflow-y: auto;
        }

        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 24px;
        }

        .header h2 {
            font-size: 24px;
            font-weight: 600;
            color: white;
            text-align: center;
            width: 100%;
            text-shadow: 2px 2px 4px rgba(0, 0, 0, 0.3);
        }

        .user-info { display: flex; align-items: center; gap: 12px; }

        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: var(--primary);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
        }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 20px;
            margin-bottom: 24px;
        }

        .stat-card {
            background: white;
            border-radius: var(--radius);
            padding: 20px;
            box-shadow: var(--shadow);
            transition: var(--transition);
            border-left: 4px solid var(--primary);
        }

        .stat-card:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-lg);
        }

        .stat-card.premium { border-left-color: var(--secondary); }
        .stat-card.claims  { border-left-color: var(--warning); }
        .stat-card.agents  { border-left-color: var(--primary); }
        .stat-card.pending { border-left-color: var(--danger); }

        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 12px;
        }

        .stat-title { font-size: 14px; color: var(--gray-500); font-weight: 500; }

        .stat-icon {
            width: 40px;
            height: 40px;
            border-radius: 8px;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
        }

        .stat-icon.premium { background: var(--secondary); }
        .stat-icon.claims  { background: var(--warning); }
        .stat-icon.agents  { background: var(--primary); }
        .stat-icon.pending { background: var(--danger); }

        .stat-value { font-size: 24px; font-weight: 700; margin-bottom: 4px; }

        .stat-change {
            display: flex;
            align-items: center;
            gap: 4px;
            font-size: 12px;
            font-weight: 500;
        }

        .stat-change.positive { color: var(--secondary); }
        .stat-change.negative { color: var(--danger); }

        /* Charts Section */
        .charts-section {
            display: grid;
            grid-template-columns: 2fr 1fr;
            gap: 20px;
            margin-bottom: 24px;
        }

        .chart-card {
            background: white;
            border-radius: var(--radius);
            padding: 20px;
            box-shadow: var(--shadow);
        }

        .chart-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
        }

        .chart-title { font-size: 16px; font-weight: 600; color: var(--gray-800); }

        .chart-actions { display: flex; gap: 8px; }

        .chart-action {
            background: var(--gray-100);
            border: none;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 12px;
            color: var(--gray-600);
            cursor: pointer;
            transition: var(--transition);
        }

        .chart-action:hover, .chart-action.active {
            background: var(--primary);
            color: white;
        }

        .chart-container { height: 300px; position: relative; }

        /* Recent Activity */
        .activity-section {
            background: white;
            border-radius: var(--radius);
            padding: 20px;
            box-shadow: var(--shadow);
        }

        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
        }

        .section-title { font-size: 18px; font-weight: 600; color: var(--gray-800); }

        .view-all {
            color: var(--primary);
            text-decoration: none;
            font-size: 14px;
            font-weight: 500;
        }

        .activity-table { width: 100%; border-collapse: collapse; }

        .activity-table th {
            text-align: left;
            padding: 12px 16px;
            border-bottom: 1px solid var(--gray-200);
            color: var(--gray-500);
            font-weight: 500;
            font-size: 14px;
        }

        .activity-table td {
            padding: 12px 16px;
            border-bottom: 1px solid var(--gray-100);
        }

        .activity-table tr:last-child td { border-bottom: none; }
        .activity-table tr:hover { background: var(--gray-50); }

        .policy-id { font-weight: 500; color: var(--primary); }

        .status-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }

        .status-active   { background: var(--primary-light); color: var(--primary); }
        .status-pending  { background: #fef3c7; color: #d97706; }
        .status-lapsed   { background: #fee2e2; color: var(--danger); }
        .status-submitted{ background: #f3e8ff; color: #7c3aed; }

        /* Responsive */
        @media (max-width: 1024px) { .charts-section { grid-template-columns: 1fr; } }
        @media (max-width: 768px)  {
            .dashboard-container { grid-template-columns: 1fr; }
            .sidebar { display: none; }
            .stats-grid { grid-template-columns: repeat(2, 1fr); }
        }
        @media (max-width: 480px) {
            .stats-grid { grid-template-columns: 1fr; }
            .header { flex-direction: column; align-items: flex-start; gap: 12px; }
        }

        /* Background Styles */
        body {
            background-image: url('images/ayo_banner4.png');
            background-size: cover;
            background-position: center;
            background-repeat: no-repeat;
            background-attachment: fixed;
        }

        body::before {
            content: "";
            position: fixed;
            top: 0; left: 0;
            width: 100%; height: 100%;
            background-image: url('images/ayo_banner4.png');
            background-size: cover;
            background-position: center;
            background-repeat: no-repeat;
            background-attachment: fixed;
            opacity: 0.5;
            z-index: -1;
        }
    </style>
</head>
<body>
    <?php include 'main_menu_app.php'; ?>

    <!-- Main Content -->
    <main class="main-content">
        <div class="header">
            <h2>Abrabo Pa Dashboard</h2>
            <div class="user-info">
                <div class="user-avatar">
                    <?php echo substr($full_namen ?? 'U', 0, 1); ?>
                </div>
                <div>
                    <div style="font-weight: 500; color: white;">
    <?php echo htmlspecialchars($full_namen ?? 'User'); ?>
</div>

<div style="font-size: 12px; color: rgba(255, 255, 255, 0.8);">
    <?php echo htmlspecialchars($role ?? 'Admin'); ?>
</div>
                </div>
            </div>
        </div>

        <!-- Stats Grid -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-header">
                    <div class="stat-title">Total Active Policies</div>
                    <div class="stat-icon agents">
                        <i class="material-icons">assignment</i>
                    </div>
                </div>
                <div class="stat-value"><?php echo number_format($total_policies); ?></div>
                <div class="stat-change positive">
                    <i class="material-icons">arrow_upward</i>
                    <span><?php echo $total_policies > 0 ? 'Active' : 'No data'; ?></span>
                </div>
            </div>

            <div class="stat-card premium">
                <div class="stat-header">
                    <div class="stat-title">Monthly Premiums</div>
                    <div class="stat-icon premium">
                        <i class="material-icons">payments</i>
                    </div>
                </div>
                <div class="stat-value">₵<?php echo number_format($monthly_premiums, 2); ?></div>
                <div class="stat-change positive">
                    <i class="material-icons">arrow_upward</i>
                    <span><?php echo $monthly_premiums > 0 ? 'Collected' : 'No premiums'; ?></span>
                </div>
            </div>

            <div class="stat-card claims">
                <div class="stat-header">
                    <div class="stat-title">Active Claims</div>
                    <div class="stat-icon claims">
                        <i class="material-icons">description</i>
                    </div>
                </div>
                <div class="stat-value"><?php echo number_format($active_claims); ?></div>
                <div class="stat-change <?php echo $active_claims > 0 ? 'negative' : 'positive'; ?>">
                    <i class="material-icons"><?php echo $active_claims > 0 ? 'arrow_upward' : 'arrow_downward'; ?></i>
                    <span><?php echo $active_claims > 0 ? 'Require attention' : 'All clear'; ?></span>
                </div>
            </div>

            <div class="stat-card pending">
                <div class="stat-header">
                    <div class="stat-title">Pending Approvals</div>
                    <div class="stat-icon pending">
                        <i class="material-icons">pending_actions</i>
                    </div>
                </div>
                <div class="stat-value"><?php echo number_format($pending_approvals); ?></div>
                <div class="stat-change <?php echo $pending_approvals > 0 ? 'positive' : 'negative'; ?>">
                    <i class="material-icons"><?php echo $pending_approvals > 0 ? 'arrow_upward' : 'arrow_downward'; ?></i>
                    <span><?php echo $pending_approvals > 0 ? 'Awaiting action' : 'All processed'; ?></span>
                </div>
            </div>
        </div>

        <!-- Charts Section -->
        <div class="charts-section">
            <div class="chart-card">
                <div class="chart-header">
                    <div class="chart-title">Premium Collection Trend</div>
                    <div class="chart-actions">
                        <button class="chart-action active" onclick="updateChart('monthly')">Monthly</button>
                        <button class="chart-action" onclick="updateChart('quarterly')">Quarterly</button>
                        <button class="chart-action" onclick="updateChart('yearly')">Yearly</button>
                    </div>
                </div>
                <div class="chart-container">
                    <canvas id="premiumChart"></canvas>
                </div>
            </div>

            <div class="chart-card">
                <div class="chart-header">
                    <div class="chart-title">Policy Distribution by Product</div>
                </div>
                <div class="chart-container">
                    <canvas id="policyChart"></canvas>
                </div>
            </div>
        </div>

        <!-- Recent Activity -->
        <div class="activity-section">
            <div class="section-header">
                <div class="section-title">Recent Policy Activity</div>
                <a href="policy_list.php" class="view-all">View All</a>
            </div>
            <table class="activity-table">
                <thead>
                    <tr>
                        <th>Policy Number</th>
                        <th>Customer</th>
                        <th>Product</th>
                        <th>Premium</th>
                        <th>Status</th>
                        <th>Date</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($recent_activity_result && $recent_activity_result->num_rows > 0): ?>
                        <?php while($row = $recent_activity_result->fetch_assoc()): ?>
                            <tr>
                                <td class="policy-id"><?php echo htmlspecialchars($row['policy_number']); ?></td>
                                <td><?php echo htmlspecialchars($row['full_name']); ?></td>
                                <td><?php echo htmlspecialchars($row['product_name']); ?></td>
                                <td>₵<?php echo number_format((float)$row['set_premium'], 2); ?></td>
                                <td>
                                    <?php 
                                    $status_class = 'status-active';
                                    $status_text = 'Active';

                                    if (($row['policy_status'] ?? '') == 'PENDING') {
                                        $status_class = 'status-pending';
                                        $status_text = 'Pending';
                                    } elseif (($row['policy_status'] ?? '') == 'LAPSED') {
                                        $status_class = 'status-lapsed';
                                        $status_text = 'Lapsed';
                                    } elseif (($row['policy_status'] ?? '') == 'SUBMITTED') {
                                        $status_class = 'status-submitted';
                                        $status_text = 'Submitted';
                                    }
                                    ?>
                                    <span class="status-badge <?php echo $status_class; ?>"><?php echo $status_text; ?></span>
                                </td>
                                <td><?php echo htmlspecialchars($row['created_date']); ?></td>
                            </tr>
                        <?php endwhile; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="6" style="text-align: center; padding: 20px; color: var(--gray-500);">
                                No recent policy activity found.
                            </td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </main>

    <script>
        // Store the chart data from PHP
        const premiumData = {
            labels: <?php echo json_encode($premium_labels); ?>,
            data: <?php echo json_encode($premium_data); ?>,
            monthly: {
                labels: <?php echo json_encode($premium_labels); ?>,
                data: <?php echo json_encode($premium_data); ?>
            },
            quarterly: {
                labels: ['Q1', 'Q2', 'Q3', 'Q4'],
                data: [0, 0, 0, 0] // change to real quarterly data if needed
            },
            yearly: {
                labels: ['2023', '2024', '2025', '2026'],
                data: [0, 0, 0, 0] // change to real yearly data if needed
            }
        };

        const policyData = {
            labels: <?php echo json_encode($policy_labels); ?>,
            data: <?php echo json_encode($policy_data); ?>,
            colors: <?php echo json_encode(array_slice(['#2563eb', '#10b981', '#f59e0b', '#8b5cf6', '#ef4444'], 0, count($policy_labels))); ?>
        };

        let premiumChart;
        let policyChart;

        document.addEventListener('DOMContentLoaded', function() {

            // Premium Collection Chart
            const premiumCtx = document.getElementById('premiumChart').getContext('2d');
            premiumChart = new Chart(premiumCtx, {
                type: 'line',
                data: {
                    labels: premiumData.monthly.labels,
                    datasets: [{
                        label: 'Premium Collection (₵)',
                        data: premiumData.monthly.data,
                        borderColor: '#2563eb',
                        backgroundColor: 'rgba(37, 99, 235, 0.1)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { display: false } },
                    scales: {
                        y: {
                            beginAtZero: true,
                            grid: { drawBorder: false },
                            ticks: {
                                callback: function(value) {
                                    if (value >= 1000000) return '₵' + (value / 1000000).toFixed(1) + 'M';
                                    if (value >= 1000) return '₵' + (value / 1000).toFixed(1) + 'K';
                                    return '₵' + value;
                                }
                            }
                        },
                        x: { grid: { display: false } }
                    }
                }
            });

            // Policy Distribution Chart
            const policyCtx = document.getElementById('policyChart').getContext('2d');
            policyChart = new Chart(policyCtx, {
                type: 'doughnut',
                data: {
                    labels: policyData.labels,
                    datasets: [{
                        data: policyData.data,
                        backgroundColor: policyData.colors,
                        borderWidth: 0
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    cutout: '70%',
                    plugins: {
                        legend: {
                            position: 'bottom',
                            labels: { boxWidth: 12, padding: 15 }
                        }
                    }
                }
            });

            // Chart period buttons style
            const chartActions = document.querySelectorAll('.chart-action');
            chartActions.forEach(button => {
                button.addEventListener('click', function() {
                    chartActions.forEach(btn => btn.classList.remove('active'));
                    this.classList.add('active');
                });
            });
        });

        function updateChart(period) {
            const chartActions = document.querySelectorAll('.chart-action');
            chartActions.forEach(btn => btn.classList.remove('active'));

            chartActions.forEach(btn => {
                if (btn.textContent.toLowerCase().includes(period)) {
                    btn.classList.add('active');
                }
            });

            if (premiumChart) {
                premiumChart.data.labels = premiumData[period].labels;
                premiumChart.data.datasets[0].data = premiumData[period].data;
                premiumChart.update();
            }
        }
    </script>
</body>
</html>

<?php
// Close database results
if (isset($recent_activity_result) && $recent_activity_result) $recent_activity_result->close();
if (isset($premium_collection_result) && $premium_collection_result) $premium_collection_result->close();
if (isset($policy_distribution_result) && $policy_distribution_result) $policy_distribution_result->close();
?>
