<?php
// Connect to Database
include("database.php");
require_once('crypto_functions.php');
include('log_functions.php');  
include('get_total_premium_functions.php');      
include("session.php");

// Get user index
$userindex = isset($_SESSION['userindex']) ? (int)$_SESSION['userindex'] : 0;

// Initialize variables
$success_message = '';
$error_message = '';

// Check for session messages
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}

// Check if we're editing an existing record
$edit_mode = false;
$existing_data = array();
$subscriber_id = 0;

// Check if ID parameter exists for editing
if (isset($_REQUEST['id']) && !empty($_REQUEST['id'])) {
    $id = mysqli_real_escape_string($con, $_REQUEST['id']);
    
    // Try to decrypt if it's encrypted
    try {
        $decrypted_id = decrypt($id);
        $subscriber_id = is_numeric($decrypted_id) ? (int)$decrypted_id : (int)$id;
    } catch (Exception $e) {
        $subscriber_id = (int)$id;
    }
    
    // If we have a valid subscriber ID, fetch existing data
    if ($subscriber_id > 0) {
        $edit_mode = true;
        
        // Prepare query using the correct column name
        $stmt = $con->prepare("
            SELECT 
                subscriptions_id,
                mobile_money_number,
                full_name,
                date_of_birth,
                gender,
                policy_number,
                product_name,
                premium,
                payment_status,
                onboarding_channel,
                type                
            FROM subscription_main
            WHERE subscriptions_id = ?
        ");
        
        if ($stmt) {
            $stmt->bind_param("i", $subscriber_id);
            $stmt->execute();
            $result_search = $stmt->get_result();
            
            if ($result_search->num_rows > 0) {
                $existing_data = $result_search->fetch_assoc();
                
                // Assign existing values to variables
                $mobile_money_number = htmlspecialchars($existing_data['mobile_money_number'] ?? '', ENT_QUOTES, 'UTF-8');
                $full_name_main = htmlspecialchars($existing_data['full_name'] ?? '', ENT_QUOTES, 'UTF-8');
                $date_of_birth = htmlspecialchars($existing_data['date_of_birth'] ?? '', ENT_QUOTES, 'UTF-8');
                $gender = htmlspecialchars($existing_data['gender'] ?? '', ENT_QUOTES, 'UTF-8');
                $policy_number = htmlspecialchars($existing_data['policy_number'] ?? '', ENT_QUOTES, 'UTF-8');
                $product_name = htmlspecialchars($existing_data['product_name'] ?? '', ENT_QUOTES, 'UTF-8');
                $premiums = htmlspecialchars($existing_data['premium'] ?? '', ENT_QUOTES, 'UTF-8');
                $payment_status = htmlspecialchars($existing_data['payment_status'] ?? '', ENT_QUOTES, 'UTF-8');
                $onboarding_channel = htmlspecialchars($existing_data['onboarding_channel'] ?? '', ENT_QUOTES, 'UTF-8');
                $type = htmlspecialchars($existing_data['type'] ?? '', ENT_QUOTES, 'UTF-8');
            }
            $stmt->close();
        }
    }
}

// Get total premium function
if ($edit_mode && isset($premiums)) {
    $data = getPremiumForAdditionalLives($con, $subscriber_id, $premiums);
    
    $count = $data['count'];
    $premium_mains = $data['premium_main'];
    $premium_one = $data['premium_one'];
    $premium_two = $data['premium_two'];
    
    if ($count == 0) {
        $total_premium = $premium_mains;
    } elseif ($count == 1) {
        $total_premium = $premium_one;
    } elseif ($count == 2) {
        $total_premium = $premium_two;
    }
    
    // Update total premium in main table
    if (isset($policy_number)) {
        $stmt = $con->prepare("
            UPDATE subscription_main
            SET total_premium = ?
            WHERE policy_number = ?
        ");
        if ($stmt) {
            $stmt->bind_param("ds", $total_premium, $policy_number);
            $stmt->execute();
            $stmt->close();
        }
    }
}

// =============================================
// HANDLE ADDING NEW ADDITIONAL LIFE
// =============================================
if (isset($_POST['add_additional_life'])) {
    // Sanitize inputs
    $full_name = trim($_POST['full_name'] ?? '');
    $mobile_money_number = trim($_POST['mobile_money_number'] ?? '');
    $date_of_birth = trim($_POST['date_of_birth'] ?? '');
    $gender = trim($_POST['gender'] ?? '');
    $relationship = trim($_POST['relationship'] ?? '');
    
    // Validate inputs
    if (empty($full_name) || empty($mobile_money_number) || empty($date_of_birth) || empty($gender) || empty($relationship)) {
        $_SESSION['error_message'] = "All fields are required!";
    } else {
        // Check if we already have 2 additional lives (maximum)
        $stmt_check = $con->prepare("SELECT COUNT(*) as count FROM subscription_additionallife WHERE subscriptions_id = ?");
        $stmt_check->bind_param("i", $subscriber_id);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();
        $row_check = $result_check->fetch_assoc();
        $stmt_check->close();
        
        if ($row_check['count'] >= 2) {
            $_SESSION['error_message'] = "Maximum of 2 additional lives allowed per subscription.";
        } else {
            // Get premium for additional life based on main product
            $additional_premium = 0;
            if (isset($product_name)) {
                $stmt_premium = $con->prepare("
                    SELECT premium_additional_life 
                    FROM cover_levels_view 
                    WHERE product_name = ? 
                    LIMIT 1
                ");
                $stmt_premium->bind_param("s", $product_name);
                $stmt_premium->execute();
                $result_premium = $stmt_premium->get_result();
                if ($row_premium = $result_premium->fetch_assoc()) {
                    $additional_premium = $row_premium['premium_additional_life'] ?? 0;
                }
                $stmt_premium->close();
            }
            
            // Insert new additional life
            $current_date = date('Y-m-d H:i:s');
            $stmt = $con->prepare("
                INSERT INTO subscription_additionallife 
                (subscriptions_id, full_name, mobile_money_number, date_of_birth, gender, relationship, premium, date)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            if ($stmt) {
                $stmt->bind_param(
                    "isssssds",
                    $subscriber_id,
                    $full_name,
                    $mobile_money_number,
                    $date_of_birth,
                    $gender,
                    $relationship,
                    $additional_premium,
                    $current_date
                );
                
                if ($stmt->execute()) {
                    $_SESSION['success_message'] = "Additional life added successfully!";
                    
                    // Log the operation
                    $data_inf = "Name: $full_name, Mobile: $mobile_money_number, DOB: $date_of_birth, Gender: $gender, Relationship: $relationship";
                    $page_path = 'subscription_amendment.php';
                    $reasons = 'Add Additional Life Successfully';
                    log_failed_login($con, $userindex, $page_path, $reasons, $data_inf);
                } else {
                    $_SESSION['error_message'] = "Error adding additional life: " . $stmt->error;
                }
                $stmt->close();
            }
        }
    }
    
    // Refresh the page
    header("Location: " . $_SERVER['PHP_SELF'] . "?id=" . urlencode(encrypt($subscriber_id)));
    exit();
}

// =============================================
// HANDLE ADDING NEW BENEFICIARY
// =============================================
if (isset($_POST['add_beneficiary'])) {
    // Sanitize inputs
    $full_name = trim($_POST['full_name'] ?? '');
    $relationship = trim($_POST['relationship'] ?? '');
    $mobile_money_number = trim($_POST['mobile_money_number'] ?? '');
    
    // Validate inputs
    if (empty($full_name) || empty($relationship) || empty($mobile_money_number)) {
        $_SESSION['error_message'] = "All fields are required!";
    } else {
        // Insert new beneficiary
        $current_date = date('Y-m-d H:i:s');
        $stmt = $con->prepare("
            INSERT INTO subscription_beneficiaries 
            (subscriptions_id, full_name, relationship, mobile_money_number, created_at)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        if ($stmt) {
            $stmt->bind_param(
                "issss",
                $subscriber_id,
                $full_name,
                $relationship,
                $mobile_money_number,
                $current_date
            );
            
            if ($stmt->execute()) {
                $_SESSION['success_message'] = "Beneficiary added successfully!";
                
                // Log the operation
                $data_inf = "Name: $full_name, Relationship: $relationship, Mobile: $mobile_money_number";
                $page_path = 'subscription_amendment.php';
                $reasons = 'Add Beneficiary Successfully';
                log_failed_login($con, $userindex, $page_path, $reasons, $data_inf);
            } else {
                $_SESSION['error_message'] = "Error adding beneficiary: " . $stmt->error;
            }
            $stmt->close();
        }
    }
    
    // Refresh the page
    header("Location: " . $_SERVER['PHP_SELF'] . "?id=" . urlencode(encrypt($subscriber_id)));
    exit();
}

// =============================================
// EXISTING CODE FOR EDITING AND REMOVING
// =============================================

// Search Additional Lifes (multiple records)
$additional_lifes = array();
if ($edit_mode && $subscriber_id > 0) {
    $stmt_additional = $con->prepare("
        SELECT 
            subscriptions_id_addtional,
            mobile_money_number,
            full_name,
            date_of_birth,
            gender,
            relationship,
            premium,
            `date`
        FROM subscription_additionallife
        WHERE subscriptions_id = ?
        ORDER BY subscriptions_id_addtional DESC
    ");
    
    if ($stmt_additional) {
        $stmt_additional->bind_param("i", $subscriber_id);
        $stmt_additional->execute();
        $result_additional = $stmt_additional->get_result();
        while ($row = $result_additional->fetch_assoc()) {
            $additional_lifes[] = $row;
        }
        $stmt_additional->close();
    }
}

// Search Subscription Beneficiaries (multiple records)
$subscription_beneficiaries = array();
if ($edit_mode && $subscriber_id > 0) {
    $stmt_beneficiaries = $con->prepare("
        SELECT 
            subscription_beneficiary_id,
            subscriptions_id,
            full_name,
            relationship,
            mobile_money_number,
            created_at
        FROM subscription_beneficiaries
        WHERE subscriptions_id = ?
        ORDER BY subscription_beneficiary_id DESC
    ");
    
    if ($stmt_beneficiaries) {
        $stmt_beneficiaries->bind_param("i", $subscriber_id);
        $stmt_beneficiaries->execute();
        $result_beneficiaries = $stmt_beneficiaries->get_result();
        while ($row = $result_beneficiaries->fetch_assoc()) {
            $subscription_beneficiaries[] = $row;
        }
        $stmt_beneficiaries->close();
    }
}

// Payment status color
$color = '';
if ($payment_status === 'Failed') {
    $color = 'red';
} elseif ($payment_status === 'Success') {
    $color = 'green';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Subscription Management | AYO Insurance</title>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    
    <style>
        :root {
            --primary-color: #2a89c9;
            --secondary-color: #6ec1e4;
            --success-color: #28a745;
            --danger-color: #dc3545;
            --warning-color: #ffc107;
            --light-bg: #f8f9fa;
            --card-shadow: 0 4px 12px rgba(0,0,0,0.08);
            --border-radius: 8px;
        }
        
        body {
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        .content {
            padding: 20px 0;
        }
        
        .main-card {
            background: white;
            border-radius: var(--border-radius);
            box-shadow: var(--card-shadow);
            margin-bottom: 25px;
            overflow: hidden;
        }
        
        .card-header-custom {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            padding: 15px 20px;
            border-bottom: none;
        }
        
        .card-header-custom h4 {
            margin: 0;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .nav-tabs-custom {
            border-bottom: 2px solid #e0e0e0;
            background: white;
            padding: 0 20px;
        }
        
        .nav-tabs-custom .nav-link {
            border: none;
            color: #666;
            font-weight: 500;
            padding: 12px 25px;
            border-radius: var(--border-radius) var(--border-radius) 0 0;
            margin-right: 5px;
            transition: all 0.3s;
            position: relative;
        }
        
        .nav-tabs-custom .nav-link:hover {
            color: var(--primary-color);
            background: #f0f7ff;
        }
        
        .nav-tabs-custom .nav-link.active {
            color: var(--primary-color);
            background: white;
            border-bottom: 3px solid var(--primary-color);
        }
        
        .tab-content-custom {
            padding: 25px;
        }
        
        .section-title {
            color: var(--primary-color);
            font-weight: 600;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 2px solid #f0f0f0;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .btn-primary-custom {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            border: none;
            color: white;
            padding: 10px 25px;
            border-radius: 6px;
            font-weight: 600;
            transition: all 0.3s;
        }
        
        .btn-primary-custom:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(42, 137, 201, 0.3);
        }
        
        .btn-success-custom {
            background: linear-gradient(135deg, #28a745, #20c997);
            border: none;
            color: white;
            padding: 10px 25px;
            border-radius: 6px;
            font-weight: 600;
            transition: all 0.3s;
        }
        
        .btn-danger-custom {
            background: linear-gradient(135deg, #dc3545, #c82333);
            border: none;
            color: white;
            padding: 8px 20px;
            border-radius: 6px;
            font-weight: 600;
            transition: all 0.3s;
        }
        
        .records-table {
            width: 100%;
            background: white;
            border-radius: var(--border-radius);
            overflow: hidden;
            box-shadow: 0 2px 8px rgba(0,0,0,0.05);
            margin-bottom: 20px;
        }
        
        .records-table thead {
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
        }
        
        .records-table th {
            padding: 15px;
            font-weight: 600;
            color: #333;
            border-bottom: 2px solid #dee2e6;
        }
        
        .records-table td {
            padding: 12px 15px;
            vertical-align: middle;
            border-bottom: 1px solid #eee;
        }
        
        .records-table tbody tr:hover {
            background-color: #f8f9fa;
        }
        
        .no-records {
            text-align: center;
            padding: 40px 20px;
            color: #6c757d;
            background: #f8f9fa;
            border-radius: var(--border-radius);
            margin-bottom: 20px;
        }
        
        .no-records i {
            font-size: 48px;
            color: #dee2e6;
            margin-bottom: 15px;
        }
        
        .action-buttons {
            display: flex;
            gap: 8px;
            flex-wrap: wrap;
        }
        
        .record-count {
            background: var(--primary-color);
            color: white;
            padding: 3px 10px;
            border-radius: 12px;
            font-size: 0.85rem;
            font-weight: 600;
            margin-left: 10px;
        }
        
        .alert-custom {
            border-radius: var(--border-radius);
            border: none;
            padding: 15px 20px;
            margin-bottom: 20px;
        }
        
        .modal-header-custom {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
        }
        
        .form-label {
            font-weight: 600;
            color: #333;
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .form-control {
            border: 1px solid #ddd;
            border-radius: 6px;
            padding: 10px 15px;
            transition: all 0.3s;
        }
        
        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(42, 137, 201, 0.25);
        }
        
        .info-box {
            background: linear-gradient(135deg, #f8fafc, #e9ecef);
            border-left: 4px solid var(--primary-color);
            border-radius: var(--border-radius);
            padding: 20px;
            margin-top: 25px;
        }
        
        .max-limit {
            background: linear-gradient(135deg, #fff3cd, #ffeaa7);
            border: 1px solid #ffc107;
            border-radius: var(--border-radius);
            padding: 15px;
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .nav-tabs-custom .nav-link {
                padding: 10px 15px;
                font-size: 0.9rem;
            }
            
            .tab-content-custom {
                padding: 15px;
            }
            
            .action-buttons {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <?php include 'main_menu_app.php'; ?>
    
    <div class="content">
        <div class="container-fluid">
            <!-- Alerts -->
            <?php if ($success_message): ?>
                <div class="alert alert-success alert-custom alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle me-2"></i>
                    <?php echo htmlspecialchars($success_message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="alert alert-danger alert-custom alert-dismissible fade show" role="alert">
                    <i class="fas fa-exclamation-circle me-2"></i>
                    <?php echo htmlspecialchars($error_message); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
            <?php endif; ?>
            
            <?php if (!$edit_mode): ?>
                <div class="alert alert-warning alert-custom">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    No subscription selected. Please select a subscription to manage.
                </div>
            <?php else: ?>
            
            <!-- Main Card -->
            <div class="main-card">
                <!-- Tabs -->
                <ul class="nav nav-tabs nav-tabs-custom" id="subscriptionTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="additional-tab" data-bs-toggle="tab" data-bs-target="#additional" type="button">
                            <i class="fas fa-users"></i> Additional Lives 
                            <span class="record-count"><?php echo count($additional_lifes); ?></span>
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="beneficiary-tab" data-bs-toggle="tab" data-bs-target="#beneficiary" type="button">
                            <i class="fas fa-user-friends"></i> Beneficiaries
                            <span class="record-count"><?php echo count($subscription_beneficiaries); ?></span>
                        </button>
                    </li>
                </ul>
                
                <!-- Tab Content -->
                <div class="tab-content" id="subscriptionTabsContent">
                    
                    <!-- Additional Lives Tab -->
                    <div class="tab-pane fade show active tab-content-custom" id="additional" role="tabpanel">
                        <div class="d-flex justify-content-between align-items-center mb-4">
                            <h5 class="section-title"><i class="fas fa-users"></i> Additional Life Records</h5>
                            <?php if (count($additional_lifes) < 2): ?>
                            <button type="button" class="btn btn-success-custom" data-bs-toggle="modal" data-bs-target="#addAdditionalModal">
                                <i class="fas fa-plus-circle"></i> Add Additional Life
                            </button>
                            <?php else: ?>
                            <span class="text-danger">
                                <i class="fas fa-exclamation-circle"></i> Maximum limit reached (2/2)
                            </span>
                            <?php endif; ?>
                        </div>
                        
                        <?php if (count($additional_lifes) >= 2): ?>
                        <div class="max-limit">
                            <i class="fas fa-info-circle me-2"></i>
                            You have reached the maximum limit of 2 additional lives per subscription.
                        </div>
                        <?php endif; ?>
                        
                        <?php if (count($additional_lifes) > 0): ?>
                            <div class="table-responsive">
                                <table class="records-table">
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Full Name</th>
                                            <th>Mobile Number</th>
                                            <th>Date of Birth</th>
                                            <th>Gender</th>
                                            <th>Relationship</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($additional_lifes as $index => $additional): ?>
                                        <tr>
                                            <td><?php echo $index + 1; ?></td>
                                            <td><?php echo htmlspecialchars($additional['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($additional['mobile_money_number']); ?></td>
                                            <td><?php echo htmlspecialchars($additional['date_of_birth']); ?></td>
                                            <td><?php echo htmlspecialchars($additional['gender']); ?></td>
                                            <td><?php echo htmlspecialchars($additional['relationship']); ?></td>
                                            <td class="action-buttons">
                                                <button type="button" class="btn btn-sm btn-primary-custom" 
                                                        onclick="editAdditionalLife(<?php echo $additional['subscriptions_id_addtional']; ?>)"
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#editAdditionalModal">
                                                    <i class="fas fa-edit"></i> Edit
                                                </button>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to remove this additional life?')">
                                                    <input type="hidden" name="additional_id" value="<?php echo $additional['subscriptions_id_addtional']; ?>">
                                                    <button type="submit" name="remove_additional" class="btn btn-sm btn-danger-custom">
                                                        <i class="fas fa-trash"></i> Remove
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="no-records">
                                <i class="fas fa-users-slash"></i>
                                <h5>No Additional Lives Found</h5>
                                <p>There are no additional lives associated with this subscription.</p>
                                <button type="button" class="btn btn-success-custom mt-3" data-bs-toggle="modal" data-bs-target="#addAdditionalModal">
                                    <i class="fas fa-plus-circle"></i> Add First Additional Life
                                </button>
                            </div>
                        <?php endif; ?>
                        
                        <div class="info-box">
                            <h6><i class="fas fa-info-circle"></i> About Additional Lives</h6>
                            <p class="mb-0">Additional lives are supplementary insurance coverage for family members or dependents linked to the main policyholder. You can add up to 2 additional lives per subscription.</p>
                        </div>
                    </div>
                    
                    <!-- Beneficiaries Tab -->
                    <div class="tab-pane fade tab-content-custom" id="beneficiary" role="tabpanel">
                        <div class="d-flex justify-content-between align-items-center mb-4">
                            <h5 class="section-title"><i class="fas fa-user-friends"></i> Subscription Beneficiary Records</h5>
                            <button type="button" class="btn btn-success-custom" data-bs-toggle="modal" data-bs-target="#addBeneficiaryModal">
                                <i class="fas fa-plus-circle"></i> Add Beneficiary
                            </button>
                        </div>
                        
                        <?php if (count($subscription_beneficiaries) > 0): ?>
                            <div class="table-responsive">
                                <table class="records-table">
                                    <thead>
                                        <tr>
                                            <th>#</th>
                                            <th>Full Name</th>
                                            <th>Relationship</th>
                                            <th>Mobile Number</th>
                                            <th>Added On</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($subscription_beneficiaries as $index => $beneficiary): ?>
                                        <tr>
                                            <td><?php echo $index + 1; ?></td>
                                            <td><?php echo htmlspecialchars($beneficiary['full_name']); ?></td>
                                            <td><?php echo htmlspecialchars($beneficiary['relationship']); ?></td>
                                            <td><?php echo htmlspecialchars($beneficiary['mobile_money_number']); ?></td>
                                            <td><?php echo htmlspecialchars($beneficiary['created_at']); ?></td>
                                            <td class="action-buttons">
                                                <button type="button" class="btn btn-sm btn-primary-custom" 
                                                        onclick="editBeneficiary(<?php echo $beneficiary['subscription_beneficiary_id']; ?>)"
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#editBeneficiaryModal">
                                                    <i class="fas fa-edit"></i> Edit
                                                </button>
                                                <form method="POST" style="display: inline;" onsubmit="return confirm('Are you sure you want to remove this beneficiary?')">
                                                    <input type="hidden" name="beneficiary_id" value="<?php echo $beneficiary['subscription_beneficiary_id']; ?>">
                                                    <button type="submit" name="remove_beneficiary" class="btn btn-sm btn-danger-custom">
                                                        <i class="fas fa-trash"></i> Remove
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="no-records">
                                <i class="fas fa-user-friends"></i>
                                <h5>No Beneficiaries Found</h5>
                                <p>There are no beneficiaries associated with this subscription.</p>
                                <button type="button" class="btn btn-success-custom mt-3" data-bs-toggle="modal" data-bs-target="#addBeneficiaryModal">
                                    <i class="fas fa-plus-circle"></i> Add First Beneficiary
                                </button>
                            </div>
                        <?php endif; ?>
                        
                        <div class="info-box">
                            <h6><i class="fas fa-info-circle"></i> About Beneficiaries</h6>
                            <p class="mb-0">Beneficiaries are individuals who will receive insurance benefits in case of an event. You can add multiple beneficiaries to a subscription.</p>
                        </div>
                    </div>
                </div>
            </div>
            
            <?php endif; ?>
        </div>
    </div>
    
    <!-- ============================================= -->
    <!-- ADDITIONAL LIVES MODALS -->
    <!-- ============================================= -->
    
    <!-- Add Additional Life Modal -->
    <div class="modal fade" id="addAdditionalModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header modal-header-custom">
                    <h5 class="modal-title"><i class="fas fa-user-plus"></i> Add Additional Life</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" action="">
                    <input type="hidden" name="subscriptions_id" value="<?php echo $subscriber_id; ?>">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-user"></i> Full Name</label>
                                    <input type="text" class="form-control" name="full_name" required maxlength="100">
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-mobile-alt"></i> Mobile Money Number</label>
                                    <input type="text" class="form-control" name="mobile_money_number" required 
                                           pattern="^\+?[0-9]{10,15}$" title="Enter valid mobile number (10-15 digits)">
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-calendar-alt"></i> Date of Birth</label>
                                    <input type="date" class="form-control" name="date_of_birth" required max="<?php echo date('Y-m-d'); ?>">
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-venus-mars"></i> Gender</label>
                                    <select class="form-control" name="gender" required>
                                        <option value="">Select Gender</option>
                                        <option value="Male">Male</option>
                                        <option value="Female">Female</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-heart"></i> Relationship</label>
                                    <select class="form-control" name="relationship" required>
                                        <option value="">Select Relationship</option>
                                        <option value="Spouse">Spouse</option>
                                        <option value="Child">Child</option>
                                        <option value="Parent">Parent</option>
                                        <option value="Sibling">Sibling</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label"><i class="fas fa-money-bill"></i> Premium</label>
                                    <input type="text" class="form-control" value="Will be calculated automatically" readonly>
                                    <small class="text-muted">Premium is based on the main product selection</small>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_additional_life" class="btn btn-success-custom">
                            <i class="fas fa-plus-circle"></i> Add Additional Life
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit Additional Life Modal (Existing) -->
    <div class="modal fade" id="editAdditionalModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header modal-header-custom">
                    <h5 class="modal-title"><i class="fas fa-edit"></i> Edit Additional Life</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" action="">
                    <div class="modal-body">
                        <input type="hidden" id="edit_additional_id" name="additional_id">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-user"></i> Full Name</label>
                                    <input type="text" class="form-control" id="edit_additional_name" name="full_name_addlife" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-mobile-alt"></i> Mobile Money Number</label>
                                    <input type="text" class="form-control" id="edit_additional_mobile" name="mobile_money_number_addlife" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-calendar-alt"></i> Date of Birth</label>
                                    <input type="date" class="form-control" id="edit_additional_dob" name="date_of_birth_addlife" required max="<?php echo date('Y-m-d'); ?>">
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-venus-mars"></i> Gender</label>
                                    <select class="form-control" id="edit_additional_gender" name="gender_addlife" required>
                                        <option value="">Select Gender</option>
                                        <option value="Male">Male</option>
                                        <option value="Female">Female</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-heart"></i> Relationship</label>
                                    <select class="form-control" id="edit_additional_relationship" name="relationship_addlife" required>
                                        <option value="">Select Relationship</option>
                                        <option value="Spouse">Spouse</option>
                                        <option value="Child">Child</option>
                                        <option value="Parent">Parent</option>
                                        <option value="Sibling">Sibling</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="submit_additional" class="btn btn-primary-custom">Update</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- ============================================= -->
    <!-- BENEFICIARIES MODALS -->
    <!-- ============================================= -->
    
    <!-- Add Beneficiary Modal -->
    <div class="modal fade" id="addBeneficiaryModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header modal-header-custom">
                    <h5 class="modal-title"><i class="fas fa-user-plus"></i> Add Beneficiary</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" action="">
                    <input type="hidden" name="subscriptions_id" value="<?php echo $subscriber_id; ?>">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-user"></i> Full Name</label>
                                    <input type="text" class="form-control" name="full_name" required maxlength="100">
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-heart"></i> Relationship</label>
                                    <select class="form-control" name="relationship" required>
                                        <option value="">Select Relationship</option>
                                        <option value="Spouse">Spouse</option>
                                        <option value="Child">Child</option>
                                        <option value="Parent">Parent</option>
                                        <option value="Sibling">Sibling</option>
                                        <option value="Friend">Friend</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-mobile-alt"></i> Mobile Money Number</label>
                                    <input type="text" class="form-control" name="mobile_money_number" required 
                                           pattern="^\+?[0-9]{10,15}$" title="Enter valid mobile number (10-15 digits)">
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label"><i class="fas fa-calendar-alt"></i> Added Date</label>
                                    <input type="text" class="form-control" value="<?php echo date('Y-m-d'); ?>" readonly>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_beneficiary" class="btn btn-success-custom">
                            <i class="fas fa-plus-circle"></i> Add Beneficiary
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit Beneficiary Modal (Existing) -->
    <div class="modal fade" id="editBeneficiaryModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header modal-header-custom">
                    <h5 class="modal-title"><i class="fas fa-edit"></i> Edit Beneficiary</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST" action="">
                    <div class="modal-body">
                        <input type="hidden" id="edit_beneficiary_id" name="beneficiary_id">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-user"></i> Full Name</label>
                                    <input type="text" class="form-control" id="edit_beneficiary_name" name="full_name_beneficiary" required>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-heart"></i> Relationship</label>
                                    <select class="form-control" id="edit_beneficiary_relationship" name="relationship_beneficiary" required>
                                        <option value="">Select Relationship</option>
                                        <option value="Spouse">Spouse</option>
                                        <option value="Child">Child</option>
                                        <option value="Parent">Parent</option>
                                        <option value="Sibling">Sibling</option>
                                        <option value="Friend">Friend</option>
                                        <option value="Other">Other</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label required"><i class="fas fa-mobile-alt"></i> Mobile Money Number</label>
                                    <input type="text" class="form-control" id="edit_beneficiary_mobile" name="mobile_money_number_beneficiary" required>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="submit_beneficiary" class="btn btn-primary-custom">Update</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Bootstrap JS Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Store data for editing
        const additionalLifeData = <?php echo json_encode($additional_lifes); ?>;
        const beneficiaryData = <?php echo json_encode($subscription_beneficiaries); ?>;
        
        // Edit Additional Life Function
        function editAdditionalLife(id) {
            const data = additionalLifeData.find(item => item.subscriptions_id_addtional == id);
            if (data) {
                document.getElementById('edit_additional_id').value = id;
                document.getElementById('edit_additional_name').value = data.full_name;
                document.getElementById('edit_additional_mobile').value = data.mobile_money_number;
                document.getElementById('edit_additional_dob').value = data.date_of_birth;
                document.getElementById('edit_additional_gender').value = data.gender;
                document.getElementById('edit_additional_relationship').value = data.relationship;
            }
        }
        
        // Edit Beneficiary Function
        function editBeneficiary(id) {
            const data = beneficiaryData.find(item => item.subscription_beneficiary_id == id);
            if (data) {
                document.getElementById('edit_beneficiary_id').value = id;
                document.getElementById('edit_beneficiary_name').value = data.full_name;
                document.getElementById('edit_beneficiary_relationship').value = data.relationship;
                document.getElementById('edit_beneficiary_mobile').value = data.mobile_money_number;
            }
        }
        
        // Form validation
        document.addEventListener('DOMContentLoaded', function() {
            // Mobile number formatting
            const mobileInputs = document.querySelectorAll('input[name*="mobile"]');
            mobileInputs.forEach(input => {
                input.addEventListener('input', function() {
                    this.value = this.value.replace(/\D/g, '');
                });
            });
            
            // Initialize the first tab as active
            const firstTab = document.querySelector('#additional-tab');
            if (firstTab) {
                firstTab.classList.add('active');
                const firstTabPane = document.querySelector('#additional');
                if (firstTabPane) {
                    firstTabPane.classList.add('show', 'active');
                }
            }
        });
    </script>
</body>
</html>
<?php
if (isset($con)) {
    mysqli_close($con);
}
?>